// @flow
'use strict';

import {call, delay, put, select} from 'redux-saga/effects';
import Raven from 'raven-js';

import {postJSON, fetchJSON} from '../lib/fetch';

import {camelizeObjectKeys} from '../lib/utils';

import {navigate, getLocation} from '../lib/location';

import campaignCreationSaga from './campaignCreationSaga';

import {
    DEFAULT_PROJECT_ID,
    MOBILE_MEDIATION_PROJECT_ID,
    selectProjectId,
    VIDEO_BLOGGER_PROJECT_ID,
    EFIR_BLOGGER_PROJECT_ID,
    GAMES_PROJECT_ID,
    ASSESSOR_PROJECT_ID,
} from '../selectors';
import {selectFieldDefsIds, selectSubmitPayload, selectStoreCreationDomainsAndBundleIds} from '../selectors/fields';
import {
    selectCurrentUserUrl,
    selectSubmitFormUrl,
    selectDashboardUrl,
    selectMobileMediationUrl,
    selectVideoBloggerUrl,
    selectEfirBloggerUrl,
    selectGamesUrl,
} from '../selectors/url';
import {selectUserData} from '../selectors/remote.ts';

import type {
    ApplicationStateT,
} from '../types/state';
import type {ProjectIdT} from '../selectors';

import {
    lockPending,
    pushError,
    receiveBackendErrors,
    scrollTo,
    serializeFields,
    unlockPending,
} from '../actions';

export const SERVER_ERROR_MSG = __('Server error');
export const CHANGE_LOGIN_ERROR_MSG = __('Your current login is different from login which you started filling out the form');

export const REDIRECT_SELECTORS: {[ProjectIdT]: (ApplicationStateT) => string} = {
    [MOBILE_MEDIATION_PROJECT_ID]: selectMobileMediationUrl,
    [VIDEO_BLOGGER_PROJECT_ID]: selectVideoBloggerUrl,
    [EFIR_BLOGGER_PROJECT_ID]: selectEfirBloggerUrl,
    [DEFAULT_PROJECT_ID]: selectDashboardUrl,
    [GAMES_PROJECT_ID]: selectGamesUrl,
    [ASSESSOR_PROJECT_ID]: selectDashboardUrl,
};

export default function* submitFormSaga(): Generator<*, *, *> {
    yield put(lockPending());

    yield put(serializeFields());

    const userUrl = yield select(selectCurrentUserUrl);
    const userResponse = yield call(fetchJSON, userUrl);
    const userDataResponse = camelizeObjectKeys(userResponse.data, true);

    const userData = yield select(selectUserData);

    const {canonicalLogin} = userData;

    if (canonicalLogin !== userDataResponse.canonicalLogin) {
        yield put(pushError(new Error(CHANGE_LOGIN_ERROR_MSG)));
        yield put(unlockPending());

        return;
    }

    const {
        canFillForm,
        hasRoleInPartner2,
    } = userDataResponse;

    if (!canFillForm && hasRoleInPartner2) {
        const {pathname} = getLocation();

        if (
            pathname.startsWith('/v2/dashboard') ||
            pathname.startsWith('/v2/mobile-mediation')
        ) {
            return;
        }

        const dashboardUrl = yield select(selectDashboardUrl);
        yield call(navigate, dashboardUrl);
        return;
    }

    const sendData = yield select(selectSubmitPayload);

    const submitFormUrl = yield select(selectSubmitFormUrl);

    const campaignCreation = yield select(selectStoreCreationDomainsAndBundleIds);

    const response = yield call(postJSON, submitFormUrl, sendData);

    if (response.ok) {
        yield call(campaignCreationSaga);

        const projectId = yield select(selectProjectId);
        const redirectUrl = yield select(REDIRECT_SELECTORS[projectId]);

        // Sending Yandex.Metrika 'SubmitClick' goal
        const counterId = '19660720';
        const counter = window[`yaCounter${counterId}`];

        // Create visit params for Yandex.Metrika
        const params = {
            new_partners_requests: {
                [canonicalLogin]: campaignCreation,
            },
        };

        if (counter) {
            counter.reachGoal('SubmitClick', params, () => {
                navigate(redirectUrl);
            });

            return;
        }

        yield call(navigate, redirectUrl);
        return;
    }

    const {data} = response;

    Raven.captureMessage('Form submit error', {
        extra:
            {
                response: {payload: data},
            },
    });

    if (!data) {
        const {error} = response;
        console.error(error);
        yield put(pushError(new Error(SERVER_ERROR_MSG)));
        yield put(unlockPending());

        return;
    }

    const {data: {
        error_description: errorDescription,
        error_message: errorMessage,
        errors = {},
    }} = response;
    const {global} = errors;

    const backendError = errorMessage || errorDescription;

    if (global) {
        for (const error of global) {
            yield put(pushError(new Error(error.description)));
        }
    } else if (backendError) {
        yield put(pushError(new Error(backendError)));
    }

    yield put(receiveBackendErrors({errors}));

    // Scrolling to first element
    const fields = yield select(selectFieldDefsIds);
    const fieldsWithErrors = Object.keys(errors);

    const firstFieldWithError = fields.find(field => fieldsWithErrors.includes(field));

    yield put(scrollTo(firstFieldWithError));
    yield delay(500);
    yield put(scrollTo(null));

    yield put(unlockPending());
}
