// @flow
'use strict';

import {call, put, select} from 'redux-saga/effects';

import {
    fetchJSON,
    postJSON,
} from '../lib/fetch';

import {selectBranchId} from '../selectors';

import {
    selectFieldDef,
    selectStoreField,
} from '../selectors/fields';

import {
    selectValidationUrl,
    selectValidationPostUrl,
} from '../selectors/url';

import {
    setFieldError,
    setFieldPending,
} from '../actions';

import {
    getValidationInfo,
    validator,
} from '../lib/validators';

import type {FieldBlurActionT} from '../actions';
import type {
    FieldMetaInfoT,
    RemoteErrorT,
    RemoteValidationInfoT,
    ValidationInfoT,
} from '../lib/validators';

export default function* validateFieldSaga(action: FieldBlurActionT): Generator<*, *, *> {
    const {payload: {id}} = action;

    yield put(setFieldError(id, null));

    const storeField = yield select(selectStoreField, id);
    const fieldDef = yield select(selectFieldDef, id);

    const branchId = yield select(selectBranchId);
    if (!branchId) {
        yield put(setFieldError(id, __('Branch is not defined')));
        return;
    }

    const metaInfo = {branchId};

    const validationInfo = getValidationInfo(storeField, fieldDef);

    yield put(setFieldPending(id, true));

    const remoteError = yield call(_validateField, storeField.$$value, validationInfo, metaInfo);

    yield put(setFieldPending(id, false));

    const validationResult = validator(storeField, fieldDef, remoteError);
    if (!validationResult.ok) {
        yield put(setFieldError(id, validationResult.error));
    }
}

function* _validateField(
    storeFieldValue: any,
    validationInfo: ValidationInfoT,
    metaInfo: FieldMetaInfoT,
): Generator<*, RemoteErrorT, *> {
    if (!validationInfo.remote) {
        return {
            errorToken: null,
            text: null,
        };
    }

    return yield call(_validateFieldRemotely, storeFieldValue, validationInfo.remote, metaInfo);
}

function* _validateFieldRemotely(
    storeFieldValue: any,
    remoteValidation: RemoteValidationInfoT,
    metaInfo: FieldMetaInfoT,
): Generator<*, RemoteErrorT, *> {
    const shouldValidate = remoteValidation.shouldValidateValue
        ? remoteValidation.shouldValidateValue(storeFieldValue)
        : false;

    if (!shouldValidate) {
        return {
            errorToken: null,
            text: null,
        };
    }

    const value = remoteValidation.transformValue
        ? remoteValidation.transformValue(storeFieldValue, metaInfo)
        : storeFieldValue;

    const {errorMessage} = remoteValidation;

    if (!value) {
        return {
            errorToken: null,
            text: errorMessage,
        };
    }

    const remoteResponse = yield call(_callRemoteValidation, remoteValidation, value);
    return _parseRemoteValidationResponse(remoteResponse, remoteValidation.errorMessage);
}

function _parseRemoteValidationResponse(response: Object, defaultErrorMessage: string): RemoteErrorT {
    const {
        data: {
            error_message: errorMessage,
            error_token: errorToken,
            is_valid: isValid,
        } = {},
    } = response;

    const text = !isValid
        ? errorMessage || defaultErrorMessage
        : '';

    return {
        errorToken,
        text,
    };
}

function* _callRemoteValidation(remoteValidation: RemoteValidationInfoT, value: Object): Object {
    if (remoteValidation.method === 'POST') {
        const validationUrl = yield select(selectValidationPostUrl, remoteValidation.id);
        return yield call(postJSON, validationUrl, value);
    }

    const validationUrl = yield select(selectValidationUrl, remoteValidation.id, String(value));
    return yield call(fetchJSON, validationUrl);
}
