// @flow
'use strict';

import {find} from 'lodash';

import {decamelizeObjectKeys} from '../lib/utils';

import {
    DEFAULT_PROJECT_ID,
    selectProjectId,
} from '.';

import type {ApplicationStateT, StoreFieldT} from '../types/state';

import type {FieldDefT} from '../types/fields';

export function selectFieldDef(state: ApplicationStateT, id: string): FieldDefT | null {
    const {remote: {branch}} = state;

    if (!branch) {
        return null;
    }

    const {fields} = branch;

    const fieldDef = find(fields, {id});

    return fieldDef ? fieldDef : null;
}

export function selectStoreField(state: ApplicationStateT, id: string): StoreFieldT | null {
    const {store} = state;

    return store[id] || null;
}

export function selectStoreFieldPending(state: ApplicationStateT, id: string): boolean | null {
    const {store} = state;
    const field = store[id];

    return field ? field.$$pending : null;
}

export function selectStoreFieldByFieldDefType(state: ApplicationStateT, type: string): StoreFieldT | null {
    const {remote: {branch}, store} = state;

    if (!branch) {
        return null;
    }

    const {fields: fieldDefs} = branch;

    const fieldDef = fieldDefs.find(elem => elem.type === type);

    if (!fieldDef) {
        return null;
    }

    return store[fieldDef.id] || null;
}

export function selectStoreCreationDomainsAndBundleIds(state: ApplicationStateT): StoreFieldT | null {
    const {store} = state;
    const {campaign_creation: campaignCreation} = store;

    if (!campaignCreation) {
        return null;
    }

    const domainsAndBundleIds = campaignCreation.$$value.map((campaign: Object) => {
        const {mode, domain, bundleId} = campaign;

        if (mode === 'app') {
            return bundleId;
        }

        return domain;
    });

    return domainsAndBundleIds;
}

export function selectFieldDefsIds(state: ApplicationStateT): Array<string> {
    const {remote: {branch}} = state;

    if (!branch) {
        return [];
    }

    const {fields: fieldDefs} = branch;

    return fieldDefs.map(fieldDef => fieldDef.id);
}

export function selectSubmitPayload(state: ApplicationStateT): Object {
    const {local} = state;
    const {countryId, branchId, serializedFields} = local;
    const projectId = selectProjectId(state);

    const payload = projectId === DEFAULT_PROJECT_ID
        ? {
            countryId,
            branchId,
            fields: serializedFields,
        }
        : {
            countryId,
            branchId,
            fields: serializedFields,
            project: projectId,
        };

    return decamelizeObjectKeys(payload, true);
}
