// @flow
'use strict';

import type {
    ApplicationStateT,
} from '../types/state';

export const OFFER_PAGE_ID: 'offer' = 'offer';
export const FORM_PAGE_ID: 'form' = 'form';

export const DEFAULT_PROJECT_ID: 'default' = 'default';
export const MOBILE_MEDIATION_PROJECT_ID: 'mobile_mediation' = 'mobile_mediation';
export const VIDEO_BLOGGER_PROJECT_ID: 'video_blogger' = 'video_blogger';
export const EFIR_BLOGGER_PROJECT_ID: 'efir_blogger' = 'efir_blogger';
export const GAMES_PROJECT_ID: 'games' = 'games';
export const ASSESSOR_PROJECT_ID: 'assessor' = 'assessor';

export type PageIdT = typeof OFFER_PAGE_ID | typeof FORM_PAGE_ID | null;

export type ProjectIdT = typeof DEFAULT_PROJECT_ID |
    typeof MOBILE_MEDIATION_PROJECT_ID |
    typeof VIDEO_BLOGGER_PROJECT_ID |
    typeof EFIR_BLOGGER_PROJECT_ID |
    typeof GAMES_PROJECT_ID |
    typeof ASSESSOR_PROJECT_ID;

const BASE_URL = '/form';

export function selectPageId(state: ApplicationStateT): PageIdT {
    const {router: {location}} = state;

    if (!location) {
        return null;
    }

    if (location.pathname.startsWith('/form/offer')) {
        return OFFER_PAGE_ID;
    }

    return FORM_PAGE_ID;
}

export function selectSubpath(state: ApplicationStateT): string | null {
    const {router: {location}} = state;

    if (!location || !location.pathname.startsWith(BASE_URL)) {
        return null;
    }

    const pathParts = location.pathname
        .substring(BASE_URL.length)
        .split('/')
        .filter(Boolean);

    return pathParts.length > 0 ? pathParts[0] : null;
}

export function selectProjectId(state: ApplicationStateT): ProjectIdT {
    const subpath = selectSubpath(state);

    if (subpath === MOBILE_MEDIATION_PROJECT_ID) {
        return MOBILE_MEDIATION_PROJECT_ID;
    }

    if (subpath === VIDEO_BLOGGER_PROJECT_ID) {
        return VIDEO_BLOGGER_PROJECT_ID;
    }

    if (subpath === EFIR_BLOGGER_PROJECT_ID) {
        return EFIR_BLOGGER_PROJECT_ID;
    }

    if (subpath === GAMES_PROJECT_ID) {
        return GAMES_PROJECT_ID;
    }

    if (subpath === ASSESSOR_PROJECT_ID) {
        return ASSESSOR_PROJECT_ID;
    }

    return DEFAULT_PROJECT_ID;
}

export function selectBranchId(state: ApplicationStateT): string | null {
    const {local: {branchId}} = state;
    return branchId || null;
}
