import React from 'react';
import {SEARCH_EMPTY_TEXT} from '../../../../containers/Fields/Fields';
import {Option} from '../Select';
import {toGroupOptions} from '../Select/withNativeControl';
import {Text} from '../Text';
import {Textinput} from '../Textinput';
import {Popup} from '../Popup';
import {Menu} from '../Menu';
import {ChangeEventHandler} from '../types';
import {filterOptions} from './filterOptions';
import {getCorrectValue} from './getCorrectValue';
import {Trigger} from './Trigger';

import css from './FilteredSelect.pcss';

export interface FilteredSelectProps {
    size: 's' | 'm';
    value?: string;
    options: Option[];
    onChange: ChangeEventHandler<HTMLSelectElement>;
}

const renderControl = false;

export function FilteredSelect(props: FilteredSelectProps) {
    const {
        size,
        value,
        options,
        onChange,
        ...otherProps
    } = props;

    const inputRef = React.createRef<HTMLElement>();
    const controlRef = React.createRef<HTMLSelectElement>();

    const [opened, setOpened] = React.useState<boolean>(false);
    const [filteredOptions, setFilteredOptions] = React.useState<Option[]>(options);
    const [inputValue, setInputValue] = React.useState<string | undefined>(value);

    const onTriggerClick = React.useCallback(() => {
        if (!opened) {
            setFilteredOptions(options);
        }
        setOpened(!opened);
    }, [opened]);

    const onClosePopup = React.useCallback(() => {
        setOpened(false);
    }, [opened]);

    const onInputBlur = React.useCallback(() => {
        if (!opened) {
            setOpened(false);
        }

        const optionValue = getCorrectValue(options, value);
        setInputValue(optionValue ? optionValue : '');
    }, [opened, value, options]);

    const onChangeMenu = React.useCallback<ChangeEventHandler<HTMLElement>>(event => {
        if (controlRef.current) {
            const value = event.target.value;
            event.target = controlRef.current;
            event.target.value = value;
        }

        onChange(event as React.ChangeEvent<HTMLSelectElement>);
        setOpened(false);
    }, [onChange, setOpened]);

    // Использовать React.useState для InputValue
    // Сбрасывать значение по unBlur
    React.useEffect(() => {
        const optionValue = getCorrectValue(options, value);
        if (optionValue !== undefined) {
            setInputValue(optionValue);
        }
    }, [value, options]);

    const onChangeInput = React.useCallback<React.ChangeEventHandler<HTMLInputElement>>(event => {
        if (event.type === 'blur') {
            // onChange у TextInput почему-то дергается и на событие blur,
            // а нам это не надо - на него есть отдельный хэндлер
            // подробности тут https://st.yandex-team.ru/LEGOSUPPORT-233
            return;
        }

        const {value: newValue} = event.target;

        const filteredOptions = filterOptions(options, newValue);

        setFilteredOptions(filteredOptions);
        if (!opened) {
            setOpened(true);
        }

        setInputValue(newValue);
    }, []);

    const onInputFocus = React.useCallback(() => {
        const filteredOptions = value ? filterOptions(options, value) : options;
        setFilteredOptions(filteredOptions);

        setOpened(true);
    }, [value, options]);

    const onInputKeyDown = React.useCallback<React.KeyboardEventHandler<HTMLInputElement>>(event => {
        if (event.key === 'Enter') {
            setOpened(false);
        }

        if (event.key === 'ArrowDown' || event.key === 'ArrowUp') {
            setOpened(true);
        }
    }, []);

    return (
        <>
            <Textinput
                className={css.input}
                view='default'
                size={size}
                value={inputValue}
                addonBefore={<Trigger
                    size={size}
                    opened={opened}
                    onClick={onTriggerClick}
                />}
                innerRef={inputRef}
                onChange={onChangeInput}
                onFocus={onInputFocus}
                onBlur={onInputBlur}
                onKeyDown={onInputKeyDown}
            />
            {renderControl && (
                <select
                    {...otherProps}
                    multiple={Array.isArray(value)}
                    tabIndex={-1}
                    value={value}
                    ref={controlRef}
                    style={{ display: 'none' }}
                >
                    {options.map(toGroupOptions)}
                </select>
            )}
            <Popup
                target='anchor'
                anchor={inputRef}
                direction={['bottom-center']}
                visible={opened}
                view='default'
                scope='inplace'
                className={css.popup}
                onClose={onClosePopup}
            >
                {filteredOptions.length > 0
                    ? <Menu
                        items={filteredOptions}
                        size={size}
                        view='default'
                        width='max'
                        onChange={onChangeMenu}
                        focused={opened}
                    />
                    : <div className={css.empty}>
                        <Text
                            typography='body-long-xl'
                        >
                            {SEARCH_EMPTY_TEXT}
                        </Text>
                    </div>
                }
            </Popup>
        </>
    );
}
