import React from 'react';
import cls from 'classnames';

import {Text} from '../Text';

import styles from './RadioCheckbox.css';

const SPACE_KEY = ' ';

export type RadioCheckboxValue = unknown;

export type RadioCheckboxValues = RadioCheckboxValue[];

export type RadioCheckboxOption = {
    value: RadioCheckboxValue;
    label: string;
}

export interface RadioCheckboxProps {
    multiselect?: boolean;
    options: RadioCheckboxOption[];
    values: RadioCheckboxValues;
    onChange?: (value: RadioCheckboxValues) => void;
}

export class RadioCheckbox extends React.Component<RadioCheckboxProps> {
    private onSelect = (value: RadioCheckboxValue, isSelected: boolean) => {
        const {
            onChange,
            multiselect,
            values,
        } = this.props;

        if (!onChange) {
            return;
        }

        let newValues;

        if (multiselect) {
            if (!isSelected) {
                newValues = [...values, value];
            } else {
                newValues = values.filter(val => val !== value);
            }
        } else {
            newValues = [value];
        }

        onChange(newValues);
    }

    private onKeyPress = (e: React.KeyboardEvent<HTMLElement>, value: RadioCheckboxValue, isSelected: boolean) => {
        if (e.key === SPACE_KEY) {
            this.onSelect(value, isSelected);
        }
    }

    private renderOption(option: RadioCheckboxOption, values: RadioCheckboxValues) {
        const {
            label,
            value,
        } = option;

        const isSelected = values.some(v => v === value);

        return (
            <div
                key={`option_${value}`}
                onMouseDown={e => { e.preventDefault(); }}
                onClick={() => { this.onSelect(value, isSelected); }}
                onKeyPress={e => { this.onKeyPress(e, value, isSelected); }}
                className={cls(styles.option, {[styles.selected]: isSelected})}
                tabIndex={1}
                role='checkbox'
                aria-checked={isSelected}
            >
                <Text
                    overflow='ellipsis'
                    typography='body-long-xl'
                >
                    {label}
                </Text>
                <div
                    className={cls(styles.icon, {[styles.visible]: isSelected})}
                />
            </div>
        );
    }

    render() {
        const {
            options,
            values,
        } = this.props;

        return (
            <div className={styles.root}>
                {
                    options.map(option => this.renderOption(option, values))
                }
            </div>
        );
    }
}
