import {Action} from 'redux-actions';
import {ValidationResultT} from '../../lib/validateSimpleJsonSchema';

import {
    simpleFormActionsTypes,
    SIMPLE_FORM_ACTIONS,
    setFieldValue, setFieldValidation, setBackendErrors, setPending,
} from '../actions/SimpleFormActions';

export interface SimpleFormState {
    isPending: boolean;
    values: Record<string, any>;
    validations: Record<string, ValidationResultT>;
}

export const INITIAL_STATE: SimpleFormState = {
    isPending: false,
    values: {},
    validations: {},
};

function onFieldValueChange(
    state: SimpleFormState,
    action: ReturnType<typeof setFieldValue>,
): SimpleFormState {
    const {values} = state;
    const {name, value} = action.payload;

    const newValues = {
        ...values,
        [name]: value,
    };

    return {
        ...state,
        values: newValues,
    };
}

function onFieldValidation(
    state: SimpleFormState,
    action: ReturnType<typeof setFieldValidation>,
): SimpleFormState {
    const {validations} = state;
    const {name, validation} = action.payload;

    const newValidations = {
        ...validations,
        [name]: validation,
    };

    return {
        ...state,
        validations: newValidations,
    };
}

function onBackendErrors(
    state: SimpleFormState,
    action: ReturnType<typeof setBackendErrors>,
): SimpleFormState {
    const {validations} = state;
    const {payload} = action;

    const backendValidations = Object.keys(payload).reduce((acc, fieldId) => {
        const firstFieldError = payload[fieldId][0];

        if (firstFieldError) {
            const error = firstFieldError.description;

            acc[fieldId] = {
                ok: false,
                error,
            };
        }
        return acc;
    }, {});

    const newValidations = {
        ...validations,
        ...backendValidations,
    };

    return {
        ...state,
        validations: newValidations,
    };
}

function onSetPending(
    state: SimpleFormState,
    action: ReturnType<typeof setPending>,
): SimpleFormState {
    const isPending = action.payload;

    return {
        ...state,
        isPending,
    };
}

export function simpleFormReducer(state = INITIAL_STATE, action: Action<any>): SimpleFormState {
    const simpleFormAction = action as simpleFormActionsTypes;

    switch (simpleFormAction.type) {
        case SIMPLE_FORM_ACTIONS.SET_FIELD_VALUE:
            return onFieldValueChange(state, simpleFormAction);
        case SIMPLE_FORM_ACTIONS.SET_FIELD_VALIDATION:
            return onFieldValidation(state, simpleFormAction);
        case SIMPLE_FORM_ACTIONS.SET_BACKEND_ERRORS:
            return onBackendErrors(state, simpleFormAction);
        case SIMPLE_FORM_ACTIONS.SET_PENDING:
            return onSetPending(state, simpleFormAction);
        default:
            return state;
    }
}
