const Raven = require('raven-js');

import {Action} from 'redux-actions';
import {SagaIterator} from 'redux-saga';
import {call, put, select, takeEvery} from 'redux-saga/effects';
import {reachGoal} from '../../lib/reachGoal';
import {validateSimpleJsonSchema} from '../../lib/validateSimpleJsonSchema';
const {
    CHANGE_LOGIN_ERROR_MSG,
    SERVER_ERROR_MSG,
    REDIRECT_SELECTORS,
} = require('../../sagas/submitFormSaga');
import {selectUserData} from '../../selectors/remote';
import {ApplicationStateT, UserDataT} from '../../types/state';
import {
    FieldValueChangePayload, setBackendErrors,
    setFieldValidation,
    setFieldValue, setPending,
    SIMPLE_FORM_ACTIONS,
} from '../actions/SimpleFormActions';
import {FieldName, RoleValue} from '../constants/fields';
import {EmptyPayloadError} from '../errors/EmptyPayloadError';
import {
    fieldSchemaSelector,
    submitDataSelector,
    fieldValueSelector,
    fieldsSelector,
    getNotEmptyFields,
} from '../selectors/simpleFormSelectors';
import {FieldData, MultiselectFieldConfig} from '../types/FieldsData';

const {
    DEFAULT_PROJECT_ID,
    MOBILE_MEDIATION_PROJECT_ID,
    selectProjectId,
    VIDEO_BLOGGER_PROJECT_ID,
    EFIR_BLOGGER_PROJECT_ID,
    GAMES_PROJECT_ID,
    ASSESSOR_PROJECT_ID,
} = require('../../selectors');

const {fetchJSON, postJSON} = require('../../lib/fetch');
const {camelizeObjectKeys} = require('../../lib/utils');
const {branchChange, pushError, RECEIVE_BRANCH_ACTION} = require('../../actions');
const {
    selectCurrentUserUrl,
    selectSubmitFormUrl,
    selectDashboardUrl,
} = require('../../selectors/url');

const {getLocation, navigate} = require('../../lib/location');

const INAPP_URL = '/v2/inapp/app/create';

// TODO: возможно поменяется
export const REDIRECT_TO_INAPP = {
    [MOBILE_MEDIATION_PROJECT_ID]: true,
    [VIDEO_BLOGGER_PROJECT_ID]: false,
    [EFIR_BLOGGER_PROJECT_ID]: false,
    [DEFAULT_PROJECT_ID]: true,
    [GAMES_PROJECT_ID]: false,
    [ASSESSOR_PROJECT_ID]: false,
};

function* getRedirectUrl(): SagaIterator {
    const state = yield select();
    const projectId = selectProjectId(state);

    if (REDIRECT_TO_INAPP[projectId]) {
        const fields: FieldData[] | undefined = fieldsSelector(state);

        const rolesField = (
            fields
                ? fields.find(field => field.name === 'roles')
                : undefined
        ) as MultiselectFieldConfig | undefined;

        if (rolesField) {
            const rolesValue = rolesField.value as string[];

            if (rolesValue && rolesValue.length === 1 && rolesValue[0] === RoleValue.APP) {
                return INAPP_URL;
            }
        }
    }

    return REDIRECT_SELECTORS[projectId]();
}

function* validateFieldSaga(name: string, value: any): SagaIterator {
    const state = yield select();
    const schema = yield call(fieldSchemaSelector, state, name);

    if (!schema) {
        // TODO Omar2002 подумать что лучше здесь сделать, может выставить дефолтный ok для поля
        return;
    }

    const validation = yield call(validateSimpleJsonSchema, value, schema);

    yield put(setFieldValidation({
        name,
        validation,
    }));
}

function* onReceiveBranch(): SagaIterator {
    const state = yield select();
    const notEmptyFieldsNames = getNotEmptyFields(state);

    // Делаем валидацию для полей с непустым значением
    for (const field of notEmptyFieldsNames) {
        const {name, value} = field;
        yield call(validateFieldSaga, name, value);
    }
}

function* onFieldChangeSaga(action: Action<FieldValueChangePayload>): SagaIterator {
    const {payload} = action;

    if (!payload) {
        throw new EmptyPayloadError('onFieldChangeSaga', action);
    }

    const {
        name,
        value,
    } = payload as FieldValueChangePayload;

    const state: ApplicationStateT = yield select();
    const currentFieldValue = fieldValueSelector(state, name);

    if (name === FieldName.country && value) {
        if (currentFieldValue !== value) {
            yield put(setFieldValue({
                name: FieldName.branch,
                value: null,
            }));
        }
    }

    if (name === FieldName.branch && value) {
        yield put(branchChange(value));
    }

    yield put(setFieldValue(payload));

    yield call(validateFieldSaga, name, value);
}

function* onSubmitSaga(): SagaIterator {
    yield put(setPending(true));

    const userUrl = yield select(selectCurrentUserUrl);
    const userResponse = yield call(fetchJSON, userUrl);
    const userDataResponse = camelizeObjectKeys(userResponse.data, true) as UserDataT;

    const userData = yield select(selectUserData);

    const {canonicalLogin} = userData;

    if (canonicalLogin !== userDataResponse.canonicalLogin) {
        yield put(pushError(new Error(CHANGE_LOGIN_ERROR_MSG)));
        yield put(setPending(false));

        return;
    }

    const {
        canFillForm,
        hasRoleInPartner2,
    } = userDataResponse;

    if (!canFillForm && hasRoleInPartner2) {
        const {pathname} = getLocation();

        if (
            pathname.startsWith('/v2/dashboard') ||
            pathname.startsWith('/v2/mobile-mediation')
        ) {
            return;
        }

        const dashboardUrl = yield select(selectDashboardUrl);
        yield call(navigate, dashboardUrl);
        return;
    }

    const sendData = yield select(submitDataSelector);

    const submitFormUrl = yield select(selectSubmitFormUrl);

    const response = yield call(postJSON, submitFormUrl, sendData);

    if (response.ok) {
        const redirectUrl = yield call(getRedirectUrl);

        try {
            const config = [{
                // Счетчик - Партнерский интерфейс 2
                id: '19660720',
            }, {
                // Счетчик - Yandex Ads
                id: '68683276',
            }];

            yield call(reachGoal, config);
            navigate(redirectUrl);
            return;
        } catch (e) {
            yield call(navigate, redirectUrl);
            return;
        }
    }

    const {data} = response;

    Raven.captureMessage('Form submit error', {
        extra:
            {
                response: {payload: data},
            },
    });

    if (!data) {
        const {error} = response;
        console.error(error);
        yield put(pushError(new Error(SERVER_ERROR_MSG)));
        yield put(setPending(false));

        return;
    }

    const {data: {
        error_description: errorDescription,
        error_message: errorMessage,
        errors = {},
    }} = response;
    const {global} = errors;

    const backendError = errorMessage || errorDescription;

    if (global) {
        for (const error of global) {
            yield put(pushError(new Error(error.description)));
        }
    } else if (backendError) {
        yield put(pushError(new Error(backendError)));
    }

    yield put(setBackendErrors(errors));
    yield put(setPending(false));
}

export function* simpleFormSaga(): SagaIterator {
    yield takeEvery(SIMPLE_FORM_ACTIONS.FIELD_VALUE_CHANGE, onFieldChangeSaga);
    yield takeEvery(SIMPLE_FORM_ACTIONS.SUBMIT, onSubmitSaga);
    yield takeEvery(RECEIVE_BRANCH_ACTION, onReceiveBranch);
}
