import {sortBy} from 'lodash';
import qs from 'qs';
import {createSelector} from 'reselect';
import {ValidationResultT} from '../../lib/validateSimpleJsonSchema';

import {FieldDefT, SchemaT} from '../../types/fields';
import {ApplicationStateT, CountryDataT, TreeDataT} from '../../types/state';
import {Option} from '../components/uikit/Select';
import {MAX_DATE} from '../constants';
import {FieldName} from '../constants/fields';
import {
    DictionaryValue,
    FieldData,
    FieldsData,
    FieldType,
    FieldValue,
    FilteredSelectFieldConfig,
} from '../types/FieldsData';
import {Lang} from '../types/Lang';
import {getValuesFromDictionary} from '../utils/getValuesFromDictionary';
import {GLOSSARY} from './simpleFormSelectorsGlossary';
const {selectProjectId} = require('../../selectors/index');

const {decamelizeObjectKeys} = require('../../lib/utils');
import {VIP_COUNTRIES} from '../../containers/Fields/Fields';

const ROLES_FIELD_ID = 'roles';
const IGNORE_BACKEND_VALUE_FIELDS: string[] = [FieldName.email, FieldName.accountantEmail];

function treeSelect(state: ApplicationStateT): TreeDataT | undefined {
    return state.remote.tree;
}

function valuesSelector(state: ApplicationStateT): Record<string, any> {
    return state.simpleForm.values;
}

export function isPendingSelector(state: ApplicationStateT): boolean {
    return state.simpleForm.isPending;
}

export function langSelector(state: ApplicationStateT): Lang {
    const locale = state.remote.user?.language;

    switch (locale) {
        case Lang.english:
            return Lang.english;
        default:
            return Lang.russian;
    }
}

export function fieldValueSelector(state: ApplicationStateT, fieldName: string): any {
    const values = valuesSelector(state);
    // Как вариант - можно сделать просто функцию без обращения к стейту тогда можно будет в реселекте использовать
    return values ? values[fieldName] : undefined;
}

const branchFieldValueSelector = (state: ApplicationStateT) => fieldValueSelector(state, FieldName.branch);
const countryFieldValueSelector = (state: ApplicationStateT) => fieldValueSelector(state, FieldName.country);

function branchFieldsSelect(state: ApplicationStateT): FieldDefT[] | undefined {
    const branchId = branchFieldValueSelector(state);
    if (!branchId) {
        return undefined;
    }

    return state.remote.branch?.fields;
}

function validationsSelector(state: ApplicationStateT): Record<string, ValidationResultT> {
    return state.simpleForm.validations;
}

export function fieldSchemaSelector(state: ApplicationStateT, fieldName: string): SchemaT | null {
    const fields = branchFieldsSelect(state);

    const field = fields?.find(({id}) => id === fieldName);

    if (!field) {
        return null;
    }

    return field.jsonSchema;
}

export function fieldErrorSelector(
    validations: Record<string, ValidationResultT>,
    fieldName: string,
): string | undefined {
    const validationState = validations?.[fieldName];

    if (!validationState) {
        return undefined;
    }

    if (validationState.ok === false && typeof validationState.error === 'string') {
        // Валидатор может возвращать  | Record<string, string> для подполей, но в простой форме мы такие не используем
        // поэтому упрощаем историю и делаем здесь явную проверку на тип и возвращаем только строку
        return validationState.error;
    }

    return undefined;
}

export const countryFieldSelect = createSelector(
    treeSelect,
    countryFieldValueSelector,
    langSelector,
    (tree, value, lang): FilteredSelectFieldConfig | undefined => {
        if (!tree) {
            return undefined;
        }

        const countriesCodes = Object.keys(tree);

        const countryValues: DictionaryValue[] = sortBy(
            countriesCodes.map(code => ({
                content: tree[code].countryName,
                value: code,
            })),
            ['content', 'value'],
        );

        let values: Option[] = countryValues;

        if (lang === Lang.russian) {
            const otherCountries = countryValues.filter(({value}) => !VIP_COUNTRIES.includes(value));

            const vipCountries = VIP_COUNTRIES
                .map(countryId => countryValues.find(({value}) => value === countryId))
                .filter(Boolean) as DictionaryValue[];

            values = [
                {
                    items: vipCountries,
                },
                {
                    items: otherCountries,
                },
            ];
        }

        return {
            type: FieldType.FilteredSelect,
            label: GLOSSARY.COUNTRY_FIELD_LABEL,
            required: true,
            name: FieldName.country,
            values,
            value,
        };
    }
);

function currentTreeItemSelect(state: ApplicationStateT): CountryDataT | undefined {
    const countryValue = fieldValueSelector(state, FieldName.country);

    if (!countryValue) {
        return undefined;
    }

    const tree = treeSelect(state);

    return tree ? tree[countryValue] : undefined;
}

export const ofertaLinkSelector = createSelector(
    currentTreeItemSelect,
    branchFieldValueSelector,
    (treeItem, currentBranch) => {
        if (!treeItem || !currentBranch) {
            return undefined;
        }

        return treeItem.branches[currentBranch].ofertaUrl;
    }
);

export const branchFieldSelect = createSelector(
    currentTreeItemSelect,
    branchFieldValueSelector,
    (treeItem, value) => {
        if (!treeItem) {
            return undefined;
        }

        const branchValues: DictionaryValue[] = treeItem.order.map(branch => ({
            content: treeItem.branches[branch].branchName,
            value: branch,
        }));

        return {
            type: FieldType.Select,
            label: GLOSSARY.BRANCH_FIELD_LABEL,
            required: true,
            name: FieldName.branch,
            values: branchValues,
            value,
        };
    },
);

function locationsSearchSelector(state: ApplicationStateT): string | undefined {
    return state.router.location?.search;
}

const queryRoleValuesSelector = createSelector(
    locationsSearchSelector,
    search => {
        if (!search) {
            return undefined;
        }

        // Пример урла /form/?roles[]=9&roles[]=27 в котором выбраны две роли
        const parsedParams: Record<string, unknown> = qs.parse(search, {ignoreQueryPrefix: true});
        const {roles} = parsedParams;

        if (Array.isArray(roles)) {
            const isStringArray = !roles.find((role: any) => typeof role !== 'string');

            if (!isStringArray) {
                return undefined;
            }

            return roles;
        }

        return undefined;
    },
);

function getFieldValue(field: FieldDefT, storedValue: FieldValue): FieldValue {
    if (storedValue !== undefined) {
        return storedValue;
    }

    const {
        id,
        type,
        value: defaultValue,
        dictionary,
    } = field;

    if (IGNORE_BACKEND_VALUE_FIELDS.includes(id)) {
        // Для email-а не пробрасываем имеющийся адрес, т.к. он был только что создан в паспорте,
        // а требуется тот адрес, на который пользователь хочет получать документы. Т.е. мы сознательно
        // заставляем пользователя вручную вводить email
        return undefined;
    }

    if (type === 'radio' && dictionary && dictionary.length > 0) {
        return dictionary[0].value;
    }

    return defaultValue;
}

export const getNotEmptyFields = createSelector(
    branchFieldsSelect,
    valuesSelector,
    (branchFields, values) => {
        if (!branchFields) {
            return [];
        }

        return branchFields.map(field => {
            const {id: name} = field;
            const value = getFieldValue(field, values[name]);

            if (value === '' || value === undefined) {
                return undefined;
            }

            return {
                name,
                value,
            };
        }).filter(Boolean) as Array<{name: string, value: any}>;
    }
);

function getRoleFieldValue(
    storedValue: FieldValue,
    defaultValue: FieldValue,
    dictionaryValues: DictionaryValue[],
    queryRoles: string[] | undefined,
): FieldValue {
    if (storedValue !== undefined) {
        return storedValue;
    }

    if (defaultValue) {
        return defaultValue;
    }

    const availableValues = dictionaryValues.map(dictionaryValue => dictionaryValue.value);
    // Для случаев, когда в урле нет ролей или они не персекаются со словарем - выбираем первое значение из словаря
    const firstAvailableValue = availableValues[0];

    if (!queryRoles || queryRoles.length === 0) {
        return [firstAvailableValue];
    }

    // забираем из урла только те роли, которые есть в словаре
    const validRoles = queryRoles.filter(role => availableValues.includes(role));

    return validRoles.length > 0 ? validRoles : [firstAvailableValue];
}

export const fieldsSelector = createSelector(
    branchFieldsSelect,
    validationsSelector,
    valuesSelector,
    ofertaLinkSelector,
    queryRoleValuesSelector,
    (branchFields, validations, values, ofertaLink, queryRoles): FieldData[] | undefined => {
        if (!branchFields || branchFields.length === 0) {
            return undefined;
        }

        const fields: FieldData[] = branchFields.map(branchField => {
            const {
                name: label = '',
                value: defaultValue,
                type,
                id: name,
                dictionary,
                hint,
                style,
            } = branchField;

            const error = fieldErrorSelector(validations, name);

            const commonParams: Pick<FieldData, 'label' | 'name' | 'required' | 'error' | 'hint' | 'style'> = {
                label,
                name,
                required: true,
                error,
                hint,
                style,
            };

            const dictionaryValues = getValuesFromDictionary(dictionary);
            const storedValue = values[name];

            const value = name === ROLES_FIELD_ID
                ? getRoleFieldValue(storedValue, defaultValue, dictionaryValues, queryRoles)
                : getFieldValue(branchField, storedValue);

            switch (type) {
                case 'input':
                    return {
                        ...commonParams,
                        type: FieldType.Input,
                        value,
                    };
                case 'date_18':
                    return {
                        ...commonParams,
                        type: FieldType.Date,
                        value: value ? value : MAX_DATE,
                    };
                case 'boolean':
                    return {
                        ...commonParams,
                        type: FieldType.Boolean,
                        value,
                    };
                case 'radio':
                    return {
                        ...commonParams,
                        type: FieldType.Select,
                        values: dictionaryValues,
                        value,
                    };
                case 'multiselect':
                    return {
                        ...commonParams,
                        type: FieldType.MultiSelect,
                        values: dictionaryValues,
                        value,
                    };
                case 'gfm_text':
                    return {
                        ...commonParams,
                        type: FieldType.Text,
                        value: null, // бэкенд требует отправку этого поля со значением null
                    };
                case 'boolean_true':
                    return {
                        ...commonParams,
                        type: FieldType.Boolean,
                        value,
                    };
                case 'oferta_agreement': {
                    return {
                        ...commonParams,
                        url: ofertaLink,
                        type: FieldType.Oferta,
                        value,
                    };
                }
                default:
                    // TODO Omar2002 сделать отправку в сентри ошибки, но не ломать интерфейс
                    // return undefined;
                    throw new Error(`Unsupported field type: '${type}'`);
            }
        });

        return fields;
    },
);

export const fieldsDataSelector = createSelector(
    countryFieldSelect,
    branchFieldSelect,
    fieldsSelector,
    (countryField, branchField, branchFields) => {
        const fields: Array<FieldData | undefined> = [];
        fields.push(countryField);
        fields.push(branchField);

        const allFields = branchFields ? fields.concat(...branchFields) : fields;

        return {
            fields: allFields.filter(Boolean) as FieldData[],
        };
    }
);

export const submitDataSelector = createSelector(
    fieldsDataSelector,
    selectProjectId,
    countryFieldSelect,
    branchFieldSelect,
    (fieldsData: FieldsData, project: string, countryField?: FieldData, branchField?: FieldData) => {
        const branchId = branchField?.value;
        const countryId = countryField?.value;

        if (!branchId || !countryId) {
            return undefined;
        }

        const fields = fieldsData.fields.reduce((acc, {name, value}) => {
            acc[name] = value;
            return acc;
        }, {});

        return decamelizeObjectKeys({
            branchId,
            countryId,
            fields,
            project,
        });
    }
);

function fieldHasNotEmptyValue(value: any, type: FieldType): boolean {
    switch (type) {
        case FieldType.Boolean:
        case FieldType.Oferta:
        case FieldType.Date:
        case FieldType.Select:
        case FieldType.Radio:
        case FieldType.Input:
            return Boolean(value);
        case FieldType.MultiSelect:
            return Array.isArray(value) && value.length > 0;
        case FieldType.Text:
            return true;
        default:
            return false;
    }
}

export const allFieldsAreValidSelector = createSelector(
    fieldsSelector,
    validationsSelector,
    (fields: FieldData[] | undefined, validations: Record<string, ValidationResultT>): boolean => {
        if (!fields) {
            return false;
        }

        const notValidItem = fields?.find(({name, value, type}) => {
            if (validations[name] !== undefined) {
                return !validations[name].ok;
            }

            return !fieldHasNotEmptyValue(value, type);
        });

        return !notValidItem;
    }
);
