// @flow
'use strict';

const {evalCondition, OperatorT} = require('../../src/lib/evalCondition');

const STORE_MOCK: {[key: string]: {$$value: string}} = {
    first_field: {
        $$value: 'first',
    },
    second_field: {
        $$value: 'second',
    },
    third_field: {
        $$value: 'third',
    },
    fourth_field: {
        $$value: 'four',
    },
    fifth_field: {
        $$value: 'five',
    },
    sixth_field: {
        $$value: 'six',
    },
    seventh_field: {
        $$value: 'seven',
    },
};

const MOCK = {
    firstField: ['first', 'one'],
    secondField: 'second',
    thirdField: ['third', 'three'],
};

function getMock(operator?: OperatorT): {[key: string]: *} {
    if (!operator) {
        return MOCK;
    }

    return {
        [operator]: MOCK,
    };
}

const NOT_OR_MOCK = {
    NOT: {
        OR: {
            firstField: ['one'],
            secondField: 'fifth',
            thirdField: ['third', 'three'],
        },
    },
};

const COMPLEX_MOCK = {
    OR: {
        firstField: ['one'],
        secondField: 'fifth',
        thirdField: ['third', 'three'],
    },
    AND: {
        fourthField: 'four',
        fifthField: 'five',
    },
    NOT: {
        OR: {
            sixthField: 'six',
            seventhField: 'eight',
        },
    },
};

const COMPLEX_MOCK_WITH_NOT = {
    NOT: {
        OR: {
            firstField: ['one'],
            secondField: 'fifth',
            thirdField: ['third', 'three'],
        },
        AND: {
            fourthField: 'four',
            fifthField: 'five',
        },
        NOT: {
            OR: {
                sixthField: 'six',
                seventhField: 'eight',
            },
        },
    },
};

describe('calculateVisibilityByDeps', () => {
    test('simple AND truly condition', () => {
        const andMock = getMock('AND');
        const result = evalCondition(STORE_MOCK, andMock);
        expect(result).toEqual(true);
    });

    test('simple AND shortcut condition', () => {
        const andShortcutMock = getMock();
        const result = evalCondition(STORE_MOCK, andShortcutMock);
        expect(result).toEqual(true);
    });

    test('simple AND falsy condition', () => {
        const store = {
            ...STORE_MOCK,
            third_field: {
                $$value: 'fourth',
            },
        };
        const andMock = getMock('AND');
        const result = evalCondition(store, andMock);
        expect(result).toEqual(false);
    });

    test('simple OR truly condition', () => {
        const store = {
            first_field: {
                $$value: 'two',
            },
            second_field: {
                $$value: 'one',
            },
            third_field: {
                $$value: 'three',
            },
        };
        const orMock = getMock('OR');
        const result = evalCondition(store, orMock);
        expect(result).toEqual(true);
    });

    test('simple OR falsy condition', () => {
        const store = {
            first_field: {
                $$value: 'five',
            },
            second_field: {
                $$value: 'two',
            },
            third_field: {
                $$value: 'four',
            },
        };
        const orMock = getMock('OR');
        const result = evalCondition(store, orMock);
        expect(result).toEqual(false);
    });

    test('simple NOT AND truly condition', () => {
        const store = {
            ...STORE_MOCK,
            third_field: {
                $$value: 'fourth',
            },
        };
        const notAndShortcutMock = getMock('NOT');
        const result = evalCondition(store, notAndShortcutMock);
        expect(result).toEqual(true);
    });

    test('simple NOT AND falsy condition', () => {
        const notAndShortcutMock = getMock('NOT');
        const result = evalCondition(STORE_MOCK, notAndShortcutMock);
        expect(result).toEqual(false);
    });

    test('simple NOT OR truly condition', () => {
        const store = {
            first_field: {
                $$value: 'first',
            },
            second_field: {
                $$value: 'two',
            },
            third_field: {
                $$value: 'four',
            },
        };
        const result = evalCondition(store, NOT_OR_MOCK);
        expect(result).toEqual(true);
    });

    test('simple NOT OR falsy condition', () => {
        const result = evalCondition(STORE_MOCK, NOT_OR_MOCK);
        expect(result).toEqual(false);
    });

    test('complex condition', () => {
        const result = evalCondition(STORE_MOCK, COMPLEX_MOCK);
        expect(result).toEqual(false);
    });

    test('complex condition with not', () => {
        const result = evalCondition(STORE_MOCK, COMPLEX_MOCK_WITH_NOT);
        expect(result).toEqual(true);
    });

    test('empty condition', () => {
        const result = evalCondition(STORE_MOCK, {});
        expect(result).toEqual(false);
    });
});
