// @flow
'use struct';

import {
    fetchJSON,
    postJSON,
    uploadFile,
} from '../../src/lib/fetch';

describe('fetchJSON()', () => {
    beforeEach(() => {
        fetch.resetMocks();
    });

    it('retrieves json with simple request', () => {
        fetch.once(JSON.stringify({attr: '12345'}));

        return fetchJSON('https://localhost.yandex.ru/api/call')
            .then(res => {
                expect(res).toEqual({
                    data: {
                        attr: '12345',
                    },
                    ok: true,
                });
                expect(fetch.mock.calls.length).toEqual(1);
                expect(fetch.mock.calls[0][0]).toEqual('https://localhost.yandex.ru/api/call');
                expect(fetch.mock.calls[0][1]).toEqual({
                    credentials: 'same-origin',
                    headers: {
                        Accept: 'application/json',
                    },
                });
            });
    });

    it('returns error on network error', () => {
        fetch.mockRejectOnce(new Error('Network failure!'));

        return fetchJSON('https://localhost.yandex.ru/api/call')
            .then(res => {
                expect(res).toEqual({
                    error: 'Network failure!',
                    ok: false,
                });
                expect(fetch.mock.calls.length).toEqual(1);
                expect(fetch.mock.calls[0][0]).toEqual('https://localhost.yandex.ru/api/call');
                expect(fetch.mock.calls[0][1]).toEqual({
                    credentials: 'same-origin',
                    headers: {
                        Accept: 'application/json',
                    },
                });
            });
    });

    it('returns error on failure', () => {
        fetch.once(JSON.stringify({attr: '12345'}),
            {
                ok: false,
                status: 503,
                statusText: 'Forbidden',
            },
        );

        return fetchJSON('https://localhost.yandex.ru/api/call')
            .then(res => {
                expect(res).toEqual({
                    data: {
                        attr: '12345',
                    },
                    error: '503 Forbidden',
                    ok: false,
                });
                expect(fetch.mock.calls.length).toEqual(1);
                expect(fetch.mock.calls[0][0]).toEqual('https://localhost.yandex.ru/api/call');
                expect(fetch.mock.calls[0][1]).toEqual({
                    credentials: 'same-origin',
                    headers: {
                        Accept: 'application/json',
                    },
                });
            });
    });

    it('retries with token when retryOnUnauthorized is true', () => {
        const headers = new Headers();
        headers.set('X-Frontend-Authorization', 'authToken');

        fetch.once(JSON.stringify({ attr: '12345' }),
            {
                ok: false,
                status: 503,
                statusText: 'Forbidden',
                headers,
            },
        );

        fetch.once(JSON.stringify({attr: '12345'}));

        return fetchJSON('https://localhost.yandex.ru/api/call', {}, true)
            .then(res => {
                expect(res).toEqual({
                    data: {
                        attr: '12345',
                    },
                    ok: true,
                });

                expect(fetch.mock.calls.length).toEqual(2);

                expect(fetch.mock.calls[0][0]).toEqual('https://localhost.yandex.ru/api/call');
                expect(fetch.mock.calls[0][1]).toEqual({
                    credentials: 'same-origin',
                    headers: {
                        Accept: 'application/json',
                    },
                });

                expect(fetch.mock.calls[1][0]).toEqual('https://localhost.yandex.ru/api/call');
                expect(fetch.mock.calls[1][1]).toEqual({
                    credentials: 'same-origin',
                    headers: {
                        Accept: 'application/json',
                        'X-Frontend-Authorization': 'authToken',
                    },
                });
            });
    });

    it('retrieves json with custom opts to fetch', () => {
        fetch.once(JSON.stringify({attr: '12345'}));

        return fetchJSON('https://localhost.yandex.ru/api/call', {
            cache: 'no-cache',
            mode: 'no-cors',
            referrer: 'no-referrer',
            headers: {
                'user-agent': 'TestUserAgent/1.0',
            },
        })
            .then(res => {
                expect(res).toEqual({
                    data: {
                        attr: '12345',
                    },
                    ok: true,
                });
                expect(fetch.mock.calls.length).toEqual(1);
                expect(fetch.mock.calls[0][0]).toEqual('https://localhost.yandex.ru/api/call');
                expect(fetch.mock.calls[0][1]).toEqual({
                    cache: 'no-cache',
                    credentials: 'same-origin',
                    mode: 'no-cors',
                    referrer: 'no-referrer',
                    headers: {
                        Accept: 'application/json',
                        'user-agent': 'TestUserAgent/1.0',
                        'X-Frontend-Authorization': 'authToken',
                    },
                });
            });
    });
});

describe('postJSON()', () => {
    beforeEach(() => {
        fetch.resetMocks();
    });

    it('sends json with POST request', () => {
        fetch.once(JSON.stringify({attr: '12345'}));

        return postJSON('https://localhost.yandex.ru/api/call', {postAttr: '67890'})
            .then(res => {
                expect(res).toEqual({
                    data: {
                        attr: '12345',
                    },
                    ok: true,
                });
                expect(fetch.mock.calls.length).toEqual(1);
                expect(fetch.mock.calls[0][0]).toEqual('https://localhost.yandex.ru/api/call');
                expect(fetch.mock.calls[0][1]).toEqual({
                    credentials: 'same-origin',
                    body: JSON.stringify({postAttr: '67890'}),
                    cache: 'no-cache',
                    method: 'POST',
                    headers: {
                        Accept: 'application/vnd.api+json',
                        'Content-Type': 'application/vnd.api+json',
                        'X-Frontend-Authorization': 'authToken',
                    },
                });
            });
    });
});

describe('uploadFile()', () => {
    beforeEach(() => {
        fetch.resetMocks();
    });

    it('sends file with POST request', () => {
        fetch.once(JSON.stringify({attr: '12345'}));

        const file = new File(['Wow!'], 'wow.txt', {type: 'text/plain'});

        return uploadFile('https://localhost.yandex.ru/api/call', file)
            .then(res => {
                expect(res).toEqual({
                    data: {
                        attr: '12345',
                    },
                    ok: true,
                });
                expect(fetch.mock.calls.length).toEqual(1);
                expect(fetch.mock.calls[0][0]).toEqual('https://localhost.yandex.ru/api/call');

                const opts = fetch.mock.calls[0][1];
                const {body, credentials, cache, method, headers} = opts;

                expect(credentials).toBe('same-origin');
                expect(cache).toBe('no-cache');
                expect(method).toBe('POST');
                expect(headers).toEqual({
                    Accept: 'application/json',
                    'X-Frontend-Authorization': 'authToken',
                });

                const sentFile = body.get('upload');

                expect(sentFile.name).toBe('wow.txt');
                expect(sentFile.type).toBe('text/plain');
            });
    });
});
