// @flow
'use strict';

import {
    getDefaultsByType,
    getFilledGroups,
    areFieldsReady,
} from '../../src/lib/fields';

const FIELD_DEFS = [
    {
        groupName: 'Контактная информация',
        hint: 'Как в паспорте.',
        id: 'cyrillic_last_name',
        jsonSchema: {
            minLength: 1,
            pattern: '^[а-яА-Я-]+$',
            type: 'string',
        },
        name: 'Фамилия',
        type: 'input',
    },
    {
        groupName: 'Контактная информация',
        hint: 'Как в паспорте.',
        id: 'cyrillic_first_name',
        jsonSchema: {
            minLength: 1,
            pattern: '^[а-яА-Я]+$',
            type: 'string',
        },
        name: 'Имя',
        type: 'input',
    },
    {
        groupName: 'Контактная информация',
        hint: 'Как в паспорте.',
        id: 'cyrillic_patronymic_name',
        jsonSchema: {
            minLength: 1,
            pattern: '^[а-яА-Я]+$',
            type: 'string',
        },
        name: 'Отчество',
        type: 'input',
    },
    {
        groupName: 'Контактная информация',
        hint: 'Как в паспорте.',
        id: 'date_of_birth',
        jsonSchema: {
            maxLength: 10,
            minLength: 10,
            pattern: '^[0-9-]+$',
            type: 'string',
        },
        name: 'Дата рождения',
        type: 'date_18',
    },
    {
        groupName: 'Контактная информация',
        hint: 'Все отчетные документы будут отправляться на этот адрес.',
        id: 'email',
        jsonSchema: {
            format: 'email',
            type: 'string',
        },
        name: 'Email',
        type: 'input',
    },
    {
        groupName: 'Контактная информация',
        hint: 'Номер телефона лица, с которым будет заключен договор. В формате +X(XXX)XXXXXXX.',
        id: 'phone',
        jsonSchema: {
            maxLength: 50,
            pattern: '^\\+.*[0-9].*$',
            type: 'string',
        },
        name: 'Телефон',
        type: 'input',
    },
    {
        groupName: 'Сайт или приложение',
        id: 'campaign_creation',
        jsonSchema: {
            type: 'null',
        },
        type: 'campaign_creation',
    },
    {
        groupName: 'Подтверждение',
        id: 'newsletter',
        jsonSchema: {
            type: 'boolean',
        },
        name: 'Получать рассылку с полезными советами и рекомендациями',
        type: 'boolean',
        value: true,
    },
    {
        groupName: 'Подтверждение',
        id: 'allow_storage_and_processing_personal_data',
        jsonSchema: {
            enum: [
                true,
            ],
            type: 'boolean',
        },
        name: 'Даю согласие на обработку и хранение моих персональных данных',
        type: 'boolean_true',
        value: 'true',
    },
    {
        groupName: 'Подтверждение',
        id: 'oferta_agreement',
        jsonSchema: {
            enum: [
                true,
            ],
            type: 'boolean',
        },
        name: 'Я прочитал полный текст оферты и принимаю условия',
        type: 'oferta_agreement',
    },
    {
        groupName: 'Подтверждение',
        id: 'gdpr_warning',
        jsonSchema: {
            type: 'null',
        },
        name: 'Если вы обязаны соответствовать требованиям Общего Регламента ЕС по Защите Данных 2016/679 от 27 апреля 2016 г. (GDPR), то к вашему сотрудничеству с Яндексом применяются положения обязательного документа Data Processing Agreement, доступного по ссылке <https://yandex.com/legal/yan_dpa_ch>',
        type: 'gfm_text',
    },
];

describe('getDefaultsByType()', () => {
    it('returns default values by field type', () => {
        expect(getDefaultsByType()).toEqual({
            attachments: [],
            campaign_creation: [{mode: 'site'}],
            mobile_app_creation: [{ bundleId: '', platform: '', storeLink: '' }],
        });
    });
});

describe('getFilledGroups()', () => {
    it('returns filled groups', () => {
        const store = {
            cyrillic_last_name: {
                $$error: null,
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: 'Abc',
                $$visible: true,
            },
            cyrillic_first_name: {
                $$error: 'Test error',
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: 'Def',
                $$visible: true,
            },
            cyrillic_patronymic_name: {
                $$error: null,
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: false,
                $$value: null,
                $$visible: true,
            },
        };

        const fieldGroups = [
            {
                name: 'Group One',
                fields: [
                    'cyrillic_last_name',
                ],
            },
            {
                name: 'Group Two',
                fields: [
                    'cyrillic_first_name',
                ],
            },
            {
                name: 'Group Three',
                fields: [
                    'cyrillic_patronymic_name',
                ],
            },
        ];

        expect(getFilledGroups(fieldGroups, store, FIELD_DEFS)).toEqual([
            {
                name: 'Group One',
                checked: true,
            },
            {
                name: 'Group Two',
                checked: false,
            },
            {
                name: 'Group Three',
                checked: false,
            },
        ]);
    });
});

describe('getFieldsSubmitReadiness()', () => {
    it('returns true for simple fields', () => {
        const store = {
            cyrillic_last_name: {
                $$error: null,
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: 'Abc',
                $$visible: true,
            },
            cyrillic_first_name: {
                $$error: null,
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: 'Def',
                $$visible: true,
            },
            cyrillic_patronymic_name: {
                $$error: null,
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: 'Ghi',
                $$visible: true,
            },
        };

        expect(areFieldsReady(store, FIELD_DEFS)).toBeTrue;
    });

    it('returns false for untouched field', () => {
        const store = {
            cyrillic_last_name: {
                $$error: null,
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: false,
                $$value: null,
                $$visible: true,
            },
        };

        expect(areFieldsReady(store, FIELD_DEFS)).toBeFalse;
    });

    it('returns false for focused field', () => {
        const store = {
            cyrillic_last_name: {
                $$error: null,
                $$focus: true,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: 'Abc',
                $$visible: true,
            },
        };

        expect(areFieldsReady(store, FIELD_DEFS)).toBeFalse;
    });

    it('returns false for field with error', () => {
        const store = {
            cyrillic_last_name: {
                $$error: 'Test error',
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: 'Abc',
                $$visible: true,
            },
        };

        expect(areFieldsReady(store, FIELD_DEFS)).toBeFalse;
    });

    it('returns false for field with array of errors', () => {
        const store = {
            cyrillic_last_name: {
                $$error: [null, 'Test error'],
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: ['Abc', 'Def'],
                $$visible: true,
            },
        };

        expect(areFieldsReady(store, FIELD_DEFS)).toBeFalse;
    });

    it('returns true for field with array for errors with no errors', () => {
        const store = {
            cyrillic_last_name: {
                $$error: [null, null],
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: ['Abc', 'Def'],
                $$visible: true,
            },
        };

        expect(areFieldsReady(store, FIELD_DEFS)).toBeTrue;
    });

    it('returns false for field with subfield errors', () => {
        const store = {
            cyrillic_last_name: {
                $$error: {subfield: 'Test error'},
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: {subfield: 'Abc'},
                $$visible: true,
            },
        };

        expect(areFieldsReady(store, FIELD_DEFS)).toBeFalse;
    });

    it('returns true for field with subfields and no errors', () => {
        const store = {
            cyrillic_last_name: {
                $$error: {},
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: {subfield: 'Abc'},
                $$visible: true,
            },
        };

        expect(areFieldsReady(store, FIELD_DEFS)).toBeFalse;
    });

    it('skips fields from SKIP_FIELD_TYPES', () => {
        const store = {
            cyrillic_last_name: {
                $$error: null,
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: 'Abc',
                $$visible: true,
            },
            gdpr_warning: {
                $$error: null,
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: false,
                $$value: null,
                $$visible: true,
            },
        };

        expect(areFieldsReady(store, FIELD_DEFS)).toBeTrue;
    });

    it('returns true for filled campaign creation', () => {
        const store = {
            campaign_creation: {
                $$error: [{}],
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$visible: true,
                $$value: [
                    {
                        mode: 'site',
                        domain: 'yandex.ru',
                        link: 'https://metrika.yandex.ru/123456',
                    },
                ],
            },
        };

        expect(areFieldsReady(store, FIELD_DEFS)).toBeTrue;
    });

    it('returns false for unfilled campaign creation', () => {
        const store = {
            campaign_creation: {
                $$error: [{}],
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: true,
                $$value: [{}],
                $$visible: true,
            },
        };

        expect(areFieldsReady(store, FIELD_DEFS)).toBeFalse;
    });
});
