// @flow
'use strict';

import {
    FIELD_CHANGE_ACTION,
    FILE_UPLOAD_FIELD_ADD_ACTION,
    FILE_UPLOAD_FIELD_DELETE_ACTION,
    FILE_UPLOAD_FIELD_FINISHED_ACTION,
    INIT_STORE_ACTION,
    RECEIVE_BACKEND_ERRORS_ACTION,
    SCROLL_TO_ACTION,
    SERIALIZE_FIELDS_ACTION,
    SET_FIELD_ERROR_ACTION,
    SET_FIELD_FOCUS_ACTION,
    SET_FIELD_PENDING_ACTION,
} from '../../src/actions';

import {composeReducers} from '../../src/lib/reducers';

import {fieldsReducer} from '../../src/reducers/fieldsReducer';

const testFieldsReducer = composeReducers(fieldsReducer);

describe('fieldsReducer', () => {
    describe('on action FIELD_CHANGE_ACTION', () => {
        it('if type `payload.value` is `string`', () => {
            const action = {
                type: FIELD_CHANGE_ACTION,
                payload: {
                    id: 'accountant_phone',
                    value: 'new_phone_number',
                },
            };

            const state = {
                store: {
                    accountant_phone: {
                        $$value: null,
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                remote: {
                    branch: {
                        fields: [{
                            groupName: 'Информация',
                            hint: 'Номер телефона вашей организации.',
                            id: 'accountant_phone',
                            name: 'Номер телефона',
                            type: 'input',
                            jsonSchema: {
                                type: 'string',
                            },
                        }],
                    },
                },
                local: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {
                    accountant_phone: {
                        $$value: 'new_phone_number',
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: true,
                        $$visible: true,
                    },
                },
                remote: {
                    branch: {
                        fields: [{
                            groupName: 'Информация',
                            hint: 'Номер телефона вашей организации.',
                            id: 'accountant_phone',
                            name: 'Номер телефона',
                            type: 'input',
                            jsonSchema: {
                                type: 'string',
                            },
                        }],
                    },
                },
                local: {},
                router: {},
                simpleForm: {},
            });
        });

        it('if `payload.id` is undefined', () => {
            const action = {
                type: FIELD_CHANGE_ACTION,
                payload: {
                    id: undefined,
                    value: 'new_phone_number',
                },
            };

            const state = {
                store: {
                    accountant_phone: {
                        $$value: null,
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                remote: {
                    branch: {
                        fields: [{
                            groupName: 'Информация',
                            hint: 'Номер телефона вашей организации.',
                            id: 'accountant_phone',
                            name: 'Номер телефона',
                            type: 'input',
                            jsonSchema: {
                                type: 'string',
                            },
                        }],
                    },
                },
                local: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState.store).toEqual(expect.objectContaining({
                accountant_phone: {
                    $$value: null,
                    $$error: null,
                    $$focus: false,
                    $$hint: null,
                    $$pending: false,
                    $$touched: false,
                    $$visible: true,
                },
            }));
        });
    });

    describe('on action FIELD_CHANGE_ACTION with dependencies field', () => {
        test('dependency field', () => {
            const action = {
                type: FIELD_CHANGE_ACTION,
                payload: {
                    id: 'organization_type',
                    value: 'Другая',
                },
            };

            const state = {
                store: {
                    organization_type: {
                        $$error: null,
                        $$focus: false,
                        $$hint: 'Правовая форма вашей организации.',
                        $$pending: false,
                        $$touched: true,
                        $$value: 'ОАО',
                        $$visible: true,
                    },
                    dependency_field: {
                        $$error: null,
                        $$focus: false,
                        $$hint: 'Правовая форма вашей организации.',
                        $$pending: false,
                        $$touched: false,
                        $$value: null,
                        $$visible: false,
                    },
                },
                remote: {
                    branch: {
                        fields: [{
                            groupName: 'Организация',
                            hint: 'Правовая форма вашей организации.',
                            id: 'organization_type',
                            name: 'Другой тип организации',
                            type: 'input',
                            jsonSchema: {
                                enum: ['ОАО', 'ЗАО', 'ООО', 'УП', 'ИП', 'Другая'],
                                type: 'string',
                            },
                        }, {
                            dependencies: {
                                organizationType: ['', 'Другая'],
                            },
                            groupName: 'Организация',
                            hint: 'Правовая форма вашей организации.',
                            id: 'dependency_field',
                            name: 'Тип организации',
                            type: 'input',
                            jsonSchema: {
                                minLength: 1,
                                type: 'string',
                            },
                        }],
                    },
                },
                local: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);
            const {store} = nextState;

            expect(store.dependency_field).toEqual(expect.objectContaining({
                $$visible: true,
            }));
        });
    });

    describe('on action INIT_STORE_ACTION', () => {
        it('if `state.remote.branch` is undefined', () => {
            const action = {
                type: INIT_STORE_ACTION,
                payload: {
                    defaults: {
                        attachments: [],
                        campaign_creation: [],
                    },
                },
            };

            const state = {
                store: {
                    accountant_phone: {
                        $$value: null,
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {},
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            });
        });

        describe('if value `state.remote.branch` is list of fields', () => {
            it('and field have not `value`', () => {
                const action = {
                    type: INIT_STORE_ACTION,
                    payload: {
                        defaults: {
                            attachments: [],
                            campaign_creation: [],
                        },
                    },
                };

                const state = {
                    remote: {
                        branch: {
                            fields: [{
                                groupName: 'Контактная информация',
                                hint: 'Как в паспорте.',
                                id: 'latin_lastname',
                                jsonSchema: {
                                    maxLength: 100,
                                    pattern: 'pattern',
                                    type: 'string',
                                },
                                name: 'Фамилия',
                                type: 'input',
                            }],
                        },
                    },
                    store: {},
                    local: {},
                    router: {},
                    simpleForm: {},
                };

                const nextState = testFieldsReducer(state, action);

                expect(nextState).toEqual({
                    remote: {
                        branch: {
                            fields: [{
                                groupName: 'Контактная информация',
                                hint: 'Как в паспорте.',
                                id: 'latin_lastname',
                                jsonSchema: {
                                    maxLength: 100,
                                    pattern: 'pattern',
                                    type: 'string',
                                },
                                name: 'Фамилия',
                                type: 'input',
                            }],
                        },
                    },
                    store: {
                        latin_lastname: {
                            $$error: null,
                            $$focus: false,
                            $$hint: 'Как в паспорте.',
                            $$pending: false,
                            $$touched: false,
                            $$value: null,
                            $$visible: true,
                        },
                    },
                    local: {},
                    router: {},
                    simpleForm: {},
                });
            });

            it('and field have `value`', () => {
                const action = {
                    type: INIT_STORE_ACTION,
                    payload: {
                        defaults: {
                            attachments: [],
                            campaign_creation: [],
                        },
                    },
                };

                const state = {
                    remote: {
                        branch: {
                            fields: [{
                                groupName: 'Подтверждение',
                                id: 'newsletter',
                                jsonSchema: {
                                    type: 'boolean',
                                },
                                name: 'Получать рассылку с полезными советами и рекомендациями',
                                type: 'boolean',
                                value: true,
                            }],
                        },
                    },
                    local: {},
                    store: {},
                    router: {},
                    simpleForm: {},
                };

                const nextState = testFieldsReducer(state, action);

                expect(nextState).toEqual({
                    remote: {
                        branch: {
                            fields: [{
                                groupName: 'Подтверждение',
                                id: 'newsletter',
                                jsonSchema: {
                                    type: 'boolean',
                                },
                                name: 'Получать рассылку с полезными советами и рекомендациями',
                                type: 'boolean',
                                value: true,
                            }],
                        },
                    },
                    local: {},
                    store: {
                        newsletter: {
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$touched: true,
                            $$value: true,
                            $$visible: true,
                        },
                    },
                    router: {},
                    simpleForm: {},
                });
            });

            it('and field have `value`', () => {
                const action = {
                    type: INIT_STORE_ACTION,
                    payload: {
                        defaults: {
                            attachments: [],
                            campaign_creation: [{mode: 'site'}],
                        },
                    },
                };

                const state = {
                    remote: {
                        branch: {
                            fields: [{
                                groupName: 'Сайт или приложение',
                                id: 'campaign_creation',
                                jsonSchema: {
                                    type: 'null',
                                },
                                type: 'campaign_creation',
                            }],
                        },
                    },
                    store: {},
                    local: {},
                    router: {},
                    simpleForm: {},
                };

                const nextState = testFieldsReducer(state, action);

                expect(nextState).toEqual({
                    remote: {
                        branch: {
                            fields: [{
                                groupName: 'Сайт или приложение',
                                id: 'campaign_creation',
                                jsonSchema: {
                                    type: 'null',
                                },
                                type: 'campaign_creation',
                            }],
                        },
                    },
                    store: {
                        campaign_creation: {
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$touched: false,
                            $$value: [{mode: 'site'}],
                            $$visible: true,
                        },
                    },
                    local: {},
                    router: {},
                    simpleForm: {},
                });
            });
        });
    });

    describe('on action SERIALIZE_FIELDS_ACTION', () => {
        it('if `branch` value is `null`', () => {
            const action = {
                type: SERIALIZE_FIELDS_ACTION,
                payload: null,
            };

            const state = {
                local: {},
                remote: {
                    branch: null,
                },
                store: {
                    cyrillic_last_name: {
                        $$value: 'Иванов',
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                    cyrillic_first_name: {
                        $$value: 'Иван',
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                    phone: {
                        $$value: '+79812345671',
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                local: {
                    serializedFields: {},
                },
                remote: {
                    branch: null,
                },
                store: {
                    cyrillic_last_name: {
                        $$value: 'Иванов',
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                    cyrillic_first_name: {
                        $$value: 'Иван',
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                    phone: {
                        $$value: '+79812345671',
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                router: {},
                simpleForm: {},
            });
        });

        describe('if `branch` value is valid,', () => {
            it('and type of fields is not key serialisers', () => {
                const action = {
                    type: SERIALIZE_FIELDS_ACTION,
                    payload: null,
                };

                const state = {
                    local: {},
                    remote: {
                        branch: {
                            fields: [
                                {
                                    id: 'cyrillic_last_name',
                                    type: 'input',
                                    groupName: 'Test',
                                    hint: 'test',
                                    name: 'test',
                                    jsonSchema: {
                                        type: 'string',
                                    },
                                },
                                {
                                    id: 'cyrillic_first_name',
                                    type: 'input',
                                    groupName: 'Test',
                                    hint: 'test',
                                    name: 'test',
                                    jsonSchema: {
                                        type: 'string',
                                    },
                                },
                                {
                                    id: 'phone',
                                    type: 'input',
                                    groupName: 'Test',
                                    hint: 'test',
                                    name: 'test',
                                    jsonSchema: {
                                        type: 'string',
                                    },
                                },
                            ],
                        },
                    },
                    store: {
                        cyrillic_last_name: {
                            $$value: 'Иванов',
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$touched: false,
                            $$visible: true,
                        },
                        cyrillic_first_name: {
                            $$value: 'Иван',
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$touched: false,
                            $$visible: true,
                        },
                        phone: {
                            $$value: '+79812345671',
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$touched: false,
                            $$visible: true,
                        },
                    },
                    router: {},
                    simpleForm: {},
                };

                const nextState = testFieldsReducer(state, action);

                expect(nextState).toEqual({
                    local: {
                        serializedFields: {
                            cyrillic_last_name: 'Иванов',
                            cyrillic_first_name: 'Иван',
                            phone: '+79812345671',
                        },
                    },
                    remote: {
                        branch: {
                            fields: [
                                {
                                    id: 'cyrillic_last_name',
                                    type: 'input',
                                    groupName: 'Test',
                                    hint: 'test',
                                    name: 'test',
                                    jsonSchema: {
                                        type: 'string',
                                    },
                                },
                                {
                                    id: 'cyrillic_first_name',
                                    type: 'input',
                                    groupName: 'Test',
                                    hint: 'test',
                                    name: 'test',
                                    jsonSchema: {
                                        type: 'string',
                                    },
                                },
                                {
                                    id: 'phone',
                                    type: 'input',
                                    groupName: 'Test',
                                    hint: 'test',
                                    name: 'test',
                                    jsonSchema: {
                                        type: 'string',
                                    },
                                },
                            ],
                        },
                    },
                    store: {
                        cyrillic_last_name: {
                            $$value: 'Иванов',
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$touched: false,
                            $$visible: true,
                        },
                        cyrillic_first_name: {
                            $$value: 'Иван',
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$touched: false,
                            $$visible: true,
                        },
                        phone: {
                            $$value: '+79812345671',
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$touched: false,
                            $$visible: true,
                        },
                    },
                    router: {},
                    simpleForm: {},
                });
            });

            it('and type of fields is key serialisers', () => {
                const action = {
                    type: SERIALIZE_FIELDS_ACTION,
                    payload: null,
                };

                const state = {
                    local: {},
                    remote: {
                        branch: {
                            fields: [
                                {
                                    id: 'date_of_birth',
                                    type: 'date_18',
                                    groupName: 'Test',
                                    hint: 'test',
                                    name: 'test',
                                    jsonSchema: {
                                        type: 'string',
                                    },
                                },
                            ],
                        },
                    },
                    store: {
                        date_of_birth: {
                            $$value: '1986-07-03T08:00:00.000Z',
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$touched: false,
                            $$visible: true,
                        },
                    },
                    router: {},
                    simpleForm: {},
                };

                const nextState = testFieldsReducer(state, action);

                expect(nextState).toEqual({
                    local: {
                        serializedFields: {
                            date_of_birth: '1986-07-03',
                        },
                    },
                    remote: {
                        branch: {
                            fields: [
                                {
                                    id: 'date_of_birth',
                                    type: 'date_18',
                                    groupName: 'Test',
                                    hint: 'test',
                                    name: 'test',
                                    jsonSchema: {
                                        type: 'string',
                                    },
                                },
                            ],
                        },
                    },
                    store: {
                        date_of_birth: {
                            $$value: '1986-07-03T08:00:00.000Z',
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$touched: false,
                            $$visible: true,
                        },
                    },
                    router: {},
                    simpleForm: {},
                });
            });
        });
    });

    describe('on action SET_FIELD_ERROR_ACTION', () => {
        it('if field value failed validation', () => {
            const action = {
                type: SET_FIELD_ERROR_ACTION,
                payload: {
                    id: 'accountant_phone',
                    error: 'Невалидное значение',
                },
            };

            const state = {
                store: {
                    accountant_phone: {
                        $$value: null,
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {
                    accountant_phone: {
                        $$value: null,
                        $$error: 'Невалидное значение',
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            });
        });
    });

    describe('on action SET_FIELD_FOCUS_ACTION', () => {
        it('if focus in field', () => {
            const action = {
                type: SET_FIELD_FOCUS_ACTION,
                payload: {
                    id: 'cyrillic_first_name',
                    focus: true,
                },
            };

            const state = {
                store: {
                    cyrillic_first_name: {
                        $$value: null,
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {
                    cyrillic_first_name: {
                        $$value: null,
                        $$error: null,
                        $$focus: true,
                        $$hint: null,
                        $$pending: false,
                        $$touched: true,
                        $$visible: true,
                    },
                },
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            });
        });
    });

    describe('on action SET_FIELD_PENDING_ACTION', () => {
        it('if value `$$pending` in field change', () => {
            const action = {
                type: SET_FIELD_PENDING_ACTION,
                payload: {
                    id: 'accountant_phone',
                    pending: true,
                },
            };

            const state = {
                store: {
                    accountant_phone: {
                        $$value: null,
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {
                    accountant_phone: {
                        $$focus: false,
                        $$touched: false,
                        $$value: null,
                        $$error: null,
                        $$hint: null,
                        $$pending: true,
                        $$visible: true,
                    },
                },
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            });
        });
    });

    describe('on action RECEIVE_BACKEND_ERRORS_ACTION', () => {
        it('if backend error belongs not to the field frontend', () => {
            const action = {
                type: RECEIVE_BACKEND_ERRORS_ACTION,
                payload: {
                    errors: {
                        not_front_field: [{description: 'error message'}],
                    },
                },
            };

            const state = {
                store: {},
                local: {},
                router: {},
                remote: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {},
                local: {},
                router: {},
                remote: {},
                simpleForm: {},
            });
        });

        it('if backend error belongs to the field frontend', () => {
            const action = {
                type: RECEIVE_BACKEND_ERRORS_ACTION,
                payload: {
                    errors: {
                        inn_ur: [{description: 'error message'}],
                    },
                },
            };

            const state = {
                store: {
                    inn_ur: {
                        $$value: null,
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {
                    inn_ur: {
                        $$error: 'error message',
                        $$value: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            });
        });

        it('if backend error is for adfox account field', () => {
            const action = {
                type: RECEIVE_BACKEND_ERRORS_ACTION,
                payload: {
                    errors: {
                        adfox_account: [{
                            description: 'error message with token',
                            error_token: 'mock_token',
                        }],
                    },
                },
            };

            const state = {
                store: {
                    adfox_account: {
                        $$value: null,
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {
                    adfox_account: {
                        $$error: {
                            text: 'error message with token',
                            errorToken: 'mock_token',
                        },
                        $$value: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$visible: true,
                    },
                },
                remote: {},
                local: {},
                router: {},
                simpleForm: {},
            });
        });
    });

    describe('on action SCROLL_TO_ACTION', () => {
        it('if value `scrollTo` is fieldId', () => {
            const action = {
                type: SCROLL_TO_ACTION,
                payload: {
                    id: 'accountant_phone',
                },
            };

            const state = {
                local: {
                    countryId: '225',
                    branchId: 'russia_ur',
                },
                remote: {},
                router: {},
                store: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                local: {
                    countryId: '225',
                    branchId: 'russia_ur',
                    scrollTo: 'accountant_phone',
                },
                remote: {},
                router: {},
                store: {},
                simpleForm: {},
            });
        });
    });

    describe('on action FILE_UPLOAD_FIELD_ADD_ACTION', () => {
        it('if partner add first file', () => {
            const action = {
                type: FILE_UPLOAD_FIELD_ADD_ACTION,
                payload: {
                    fieldId: 'ag_ph_scan',
                    fileId: '00000001',
                    file: {name: 'test_file_1'},
                },
            };

            const state = {
                store: {
                    ag_ph_scan: {
                        $$touched: false,
                        $$value: [],
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$visible: true,
                    },
                },
                local: {},
                router: {},
                remote: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {
                    ag_ph_scan: {
                        $$touched: true,
                        $$value: [{
                            id: '00000001',
                            file: {name: 'test_file_1'},
                            pending: true,
                            ok: false,
                        }],
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$visible: true,
                    },
                },
                local: {},
                router: {},
                remote: {},
                simpleForm: {},
            });
        });

        it('if partner add second file', () => {
            const action = {
                type: FILE_UPLOAD_FIELD_ADD_ACTION,
                payload: {
                    fieldId: 'ag_ph_scan',
                    fileId: '00000002',
                    file: {name: 'test_file_2'},
                },
            };

            const state = {
                store: {
                    ag_ph_scan: {
                        $$touched: true,
                        $$value: [{
                            id: '00000001',
                            file: {name: 'test_file_1'},
                            pending: true,
                            ok: false,
                        }],
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$visible: true,
                    },
                },
                local: {},
                remote: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {
                    ag_ph_scan: {
                        $$touched: true,
                        $$value: [
                            {
                                id: '00000001',
                                file: {name: 'test_file_1'},
                                pending: true,
                                ok: false,
                            },
                            {
                                id: '00000002',
                                file: {name: 'test_file_2'},
                                pending: true,
                                ok: false,
                            },
                        ],
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$visible: true,
                    },
                },
                local: {},
                remote: {},
                router: {},
                simpleForm: {},
            });
        });
    });

    describe('on action FILE_UPLOAD_FIELD_DELETE_ACTION', () => {
        it('if partner have two added file and delete one', () => {
            const action = {
                type: FILE_UPLOAD_FIELD_DELETE_ACTION,
                payload: {
                    fieldId: 'ag_ph_scan',
                    fileId: '00000002',
                },
            };

            const state = {
                store: {
                    ag_ph_scan: {
                        $$touched: false,
                        $$value: [
                            {
                                id: '00000001',
                                file: {name: 'test_file_1'},
                                pending: true,
                                ok: false,
                            },
                            {
                                id: '00000002',
                                file: {name: 'test_file_2'},
                                pending: true,
                                ok: false,
                            },
                        ],
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$visible: true,
                    },
                },
                local: {},
                remote: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {
                    ag_ph_scan: {
                        $$touched: true,
                        $$value: [
                            {
                                id: '00000001',
                                file: {name: 'test_file_1'},
                                pending: true,
                                ok: false,
                            },
                        ],
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$visible: true,
                    },
                },
                local: {},
                remote: {},
                router: {},
                simpleForm: {},
            });
        });

        it('if partner delete last file', () => {
            const action = {
                type: FILE_UPLOAD_FIELD_DELETE_ACTION,
                payload: {
                    fieldId: 'ag_ph_scan',
                    fileId: '00000001',
                },
            };

            const state = {
                store: {
                    ag_ph_scan: {
                        $$touched: false,
                        $$value: [{
                            id: '00000001',
                            file: {name: 'test_file_1'},
                            pending: true,
                            ok: false,
                        }],
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$visible: true,
                    },
                },
                local: {},
                remote: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {
                    ag_ph_scan: {
                        $$touched: true,
                        $$value: [],
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$visible: true,
                    },
                },
                local: {},
                remote: {},
                router: {},
                simpleForm: {},
            });
        });
    });

    describe('on action FILE_UPLOAD_FIELD_FINISHED_ACTION', () => {
        it('if value `action.payload.ok` is `false`', () => {
            const action = {
                type: FILE_UPLOAD_FIELD_FINISHED_ACTION,
                payload: {
                    fieldId: 'ag_ph_scan',
                    fileId: '00000002',
                    ok: false,
                    key: 'key_second_file',
                },
            };

            const state = {
                store: {
                    ag_ph_scan: {
                        $$value: [{
                            id: '00000001',
                            file: {name: 'test_file_1'},
                            pending: true,
                            ok: false,
                        }],
                        $$touched: true,
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$visible: true,
                    },
                },
                local: {},
                remote: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testFieldsReducer(state, action);

            expect(nextState).toEqual({
                store: {
                    ag_ph_scan: {
                        $$value: [{
                            id: '00000001',
                            file: {name: 'test_file_1'},
                            pending: true,
                            ok: false,
                        }],
                        $$touched: true,
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$visible: true,
                    },
                },
                local: {},
                remote: {},
                router: {},
                simpleForm: {},
            });
        });

        describe('if value `action.payload.ok` is `true`', () => {
            it('and action fileId equals `$$value.id`', () => {
                const action = {
                    type: FILE_UPLOAD_FIELD_FINISHED_ACTION,
                    payload: {
                        fieldId: 'ag_ph_scan',
                        fileId: '00000001',
                        ok: true,
                        key: 'key_first_file',
                    },
                };

                const state = {
                    store: {
                        ag_ph_scan: {
                            $$value: [{
                                id: '00000001',
                                file: {name: 'test_file_1'},
                                pending: true,
                                ok: false,
                            }],
                            $$touched: true,
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$visible: true,
                        },
                    },
                    local: {},
                    remote: {},
                    router: {},
                    simpleForm: {},
                };

                const nextState = testFieldsReducer(state, action);

                expect(nextState).toEqual({
                    store: {
                        ag_ph_scan: {
                            $$value: [{
                                id: '00000001',
                                file: {name: 'test_file_1'},
                                pending: false,
                                ok: true,
                                key: 'key_first_file',
                            }],
                            $$touched: true,
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$visible: true,
                        },
                    },
                    local: {},
                    remote: {},
                    router: {},
                    simpleForm: {},
                });
            });

            it('and action fileId is not equal to `$$value.id`', () => {
                const action = {
                    type: FILE_UPLOAD_FIELD_FINISHED_ACTION,
                    payload: {
                        fieldId: 'ag_ph_scan',
                        fileId: '00000002',
                        ok: true,
                        key: 'key_second_file',
                    },
                };

                const state = {
                    store: {
                        ag_ph_scan: {
                            $$value: [{
                                id: '00000001',
                                file: {name: 'test_file_1'},
                                pending: true,
                                ok: false,
                            }],
                            $$touched: true,
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$visible: true,
                        },
                    },
                    local: {},
                    remote: {},
                    router: {},
                    simpleForm: {},
                };

                const nextState = testFieldsReducer(state, action);

                expect(nextState).toEqual({
                    store: {
                        ag_ph_scan: {
                            $$value: [{
                                id: '00000001',
                                file: {name: 'test_file_1'},
                                pending: true,
                                ok: false,
                            }],
                            $$touched: true,
                            $$error: null,
                            $$focus: false,
                            $$hint: null,
                            $$pending: false,
                            $$visible: true,
                        },
                    },
                    local: {},
                    remote: {},
                    router: {},
                    simpleForm: {},
                });
            });
        });
    });
});
