// @flow
'use strict';

import {
    BRANCH_CHANGE_ACTION,
    COUNTRY_CHANGE_ACTION,
    RECEIVE_BRANCH_ACTION,
    RECEIVE_TREE_ACTION,
    RECEIVE_USER_DATA_ACTION,
} from '../../src/actions';

import {composeReducers} from '../../src/lib/reducers';

import {treeReducer} from '../../src/reducers/treeReducer';

const testTreeReducer = composeReducers(treeReducer);

describe('treeReducer', () => {
    describe('on action BRANCH_CHANGE_ACTION', () => {
        it('if value `branchId` is `null`', () => {
            const action = {
                type: BRANCH_CHANGE_ACTION,
                payload: {
                    branchId: null,
                },
            };

            const state = {
                local: {
                    countryId: '149',
                },
                router: {},
                store: {},
                remote: {},
                simpleForm: {},
            };

            const nextState = testTreeReducer(state, action);

            expect(nextState).toEqual({
                local: {
                    countryId: '149',
                    branchId: null,
                },
                router: {},
                store: {},
                remote: {},
                simpleForm: {},
            });
        });

        it('if value `branchId` is `ag_ph`', () => {
            const action = {
                type: BRANCH_CHANGE_ACTION,
                payload: {
                    branchId: 'ag_ph',
                },
            };

            const state = {
                local: {
                    countryId: '149',
                },
                router: {},
                store: {},
                remote: {},
                simpleForm: {},
            };

            const nextState = testTreeReducer(state, action);

            expect(nextState).toEqual({
                local: {
                    countryId: '149',
                    branchId: 'ag_ph',
                },
                router: {},
                store: {},
                remote: {},
                simpleForm: {},
            });
        });
    });

    describe('on action COUNTRY_CHANGE_ACTION', () => {
        it('if value `countryId` is `149`', () => {
            const action = {
                type: COUNTRY_CHANGE_ACTION,
                payload: {
                    countryId: '149',
                },
            };

            const state = {
                local: {},
                router: {},
                store: {},
                remote: {},
                simpleForm: {},
            };

            const nextState = testTreeReducer(state, action);

            expect(nextState).toEqual({
                local: {
                    countryId: '149',
                },
                router: {},
                store: {},
                remote: {},
                simpleForm: {},
            });
        });
    });

    describe('on action RECEIVE_BRANCH_ACTION', () => {
        it('if action.payload is undefined', () => {
            const action = {
                type: RECEIVE_BRANCH_ACTION,
                payload: null,
            };

            const state = {
                local: {
                    countryId: '149',
                },
                remote: {
                    tree: {
                        '84': {
                            countryName: 'США',
                            branches: {},
                            order: ['ag_ph', 'ag_ur'],
                        },
                    },
                },
                router: {},
                store: {},
                simpleForm: {},
            };

            const nextState = testTreeReducer(state, action);

            expect(nextState).toEqual({
                local: {
                    countryId: '149',
                    fieldGroups: null,
                },
                remote: {
                    tree: {
                        '84': {
                            countryName: 'США',
                            branches: {},
                            order: ['ag_ph', 'ag_ur'],
                        },
                    },
                    branch: null,
                },
                router: {},
                store: {},
                simpleForm: {},
            });
        });

        it('if value `action.payload` is valid', () => {
            const action = {
                type: RECEIVE_BRANCH_ACTION,
                payload: {
                    data: {
                        fields: [
                            {
                                group_name: 'Контактная информация',
                                id: 'cyrillic_last_name',
                            },
                            {
                                group_name: 'Контактная информация',
                                id: 'cyrillic_first_name',
                            },
                        ],
                    },
                },
            };

            const state = {
                local: {},
                remote: {},
                store: {},
                router: {},
                simpleForm: {},
            };

            const nextState = testTreeReducer(state, action);

            expect(nextState).toEqual({
                local: {
                    fieldGroups: [{
                        name: 'Контактная информация',
                        fields: ['cyrillic_last_name', 'cyrillic_first_name'],
                    }],
                },
                remote: {
                    branch: {
                        fields: [
                            {
                                groupName: 'Контактная информация',
                                id: 'cyrillic_last_name',
                            },
                            {
                                groupName: 'Контактная информация',
                                id: 'cyrillic_first_name',
                            },
                        ],
                    },
                },
                router: {},
                store: {},
                simpleForm: {},
            });
        });
    });

    describe('on action RECEIVE_TREE_ACTION', () => {
        it('if value `action.payload` is valid', () => {
            const action = {
                type: RECEIVE_TREE_ACTION,
                payload: {
                    data: {
                        '93': {
                            country_name: 'Аргентина',
                            branches: {
                                ag_ph: {
                                    branch_name: 'Физ лицо',
                                    oferta: true,
                                },
                            },
                        },
                    },
                },
            };

            const state = {
                remote: {},
                local: {},
                router: {},
                store: {},
                simpleForm: {},
            };

            const nextState = testTreeReducer(state, action);

            expect(nextState).toEqual({
                remote: {
                    tree: {
                        '93': {
                            countryName: 'Аргентина',
                            branches: {
                                ag_ph: {
                                    branchName: 'Физ лицо',
                                    oferta: true,
                                },
                            },
                        },
                    },
                },
                local: {},
                router: {},
                store: {},
                simpleForm: {},
            });
        });
    });

    describe('on action RECEIVE_USER_DATA_ACTION', () => {
        it('if value `action.payload` is `data`', () => {
            const action = {
                type: RECEIVE_USER_DATA_ACTION,
                payload: {
                    data: {
                        can_fill_form: true,
                        forse: null,
                        is_logged: true,
                    },
                },
            };

            const state = {
                remote: {},
                local: {},
                router: {},
                store: {},
                simpleForm: {},
            };

            const nextState = testTreeReducer(state, action);

            expect(nextState).toEqual({
                remote: {
                    user: {
                        canFillForm: true,
                        forse: null,
                        isLogged: true,
                    },
                },
                local: {},
                router: {},
                store: {},
                simpleForm: {},
            });
        });

        it('if value `action.payload` is obj with key `is_logged`', () => {
            const action = {
                type: RECEIVE_USER_DATA_ACTION,
                payload: {
                    is_logged: false,
                },
            };

            const state = {
                remote: {},
                local: {},
                router: {},
                store: {},
                simpleForm: {},
            };

            const nextState = testTreeReducer(state, action);

            expect(nextState).toEqual({
                remote: {
                    user: undefined,
                },
                local: {},
                router: {},
                store: {},
                simpleForm: {},
            });
        });
    });
});
