// @flow
'use strict';

import {
    selectFieldDef,
    selectStoreField,
    selectStoreFieldByFieldDefType,
    selectFieldDefsIds,
    selectSubmitPayload,
} from '../../src/selectors/fields';

describe('selectFieldDef()', () => {
    it('returns `null` when value `branch` is undefined', () => {
        const state = {
            store: {},
            remote: {},
            local: {},
            router: {},
            simpleForm: {},
        };

        const id = 'latin_lastname';

        expect(selectFieldDef(state, id)).toBe(null);
    });

    describe('if value `branch` is valid,', () => {
        it('extracts from `fields` a field by `id`', () => {
            const state = {
                remote: {
                    branch: {
                        fields: [{
                            groupName: 'Контактная информация',
                            hint: 'Как в паспорте.',
                            id: 'latin_lastname',
                            jsonSchema: {
                                maxLength: 100,
                                pattern: 'pattern',
                                type: 'string',
                            },
                            name: 'Фамилия',
                            type: 'input',
                        }],
                    },
                },
                store: {},
                local: {},
                router: {},
                simpleForm: {},
            };

            const id = 'latin_lastname';

            expect(selectFieldDef(state, id)).toEqual({
                groupName: 'Контактная информация',
                hint: 'Как в паспорте.',
                id: 'latin_lastname',
                jsonSchema: {
                    maxLength: 100,
                    pattern: 'pattern',
                    type: 'string',
                },
                name: 'Фамилия',
                type: 'input',
            });
        });

        it('returns `null` because value `id` does not coincides with one of the fields `id`', () => {
            const state = {
                remote: {
                    branch: {
                        fields: [{
                            groupName: 'Контактная информация',
                            hint: 'Как в паспорте.',
                            id: 'latin_lastname',
                            jsonSchema: {
                                maxLength: 100,
                                pattern: 'pattern',
                                type: 'string',
                            },
                            name: 'Фамилия',
                            type: 'input',
                        }],
                    },
                },
                store: {},
                local: {},
                router: {},
                simpleForm: {},
            };

            const id = 'email';

            expect(selectFieldDef(state, id)).toBe(null);
        });
    });
});

describe('selectStoreField()', () => {
    it('extracts from `store` field description object by `id`', () => {
        const state = {
            store: {
                newsletter: {
                    $$error: null,
                    $$focus: false,
                    $$hint: null,
                    $$pending: false,
                    $$touched: true,
                    $$value: true,
                    $$visible: true,
                },
            },
            remote: {},
            local: {},
            router: {},
            simpleForm: {},
        };

        const id = 'newsletter';

        expect(selectStoreField(state, id)).toEqual({
            $$error: null,
            $$focus: false,
            $$hint: null,
            $$pending: false,
            $$touched: true,
            $$value: true,
            $$visible: true,
        });
    });

    it('if value `id` does not coincides with one of the `store`, returns value is `null`', () => {
        const state = {
            store: {
                newsletter: {
                    $$error: null,
                    $$focus: false,
                    $$hint: null,
                    $$pending: false,
                    $$touched: true,
                    $$value: true,
                    $$visible: true,
                },
            },
            remote: {},
            local: {},
            router: {},
            simpleForm: {},
        };

        const id = 'latin_lastname';

        expect(selectStoreField(state, id)).toBe(null);
    });
});

describe('selectStoreFieldByFieldDefType()', () => {
    it('returns `null` when value `branch` is undefined', () => {
        const state = {
            store: {},
            remote: {},
            local: {},
            router: {},
            simpleForm: {},
        };

        const type = 'input';

        expect(selectStoreFieldByFieldDefType(state, type)).toBe(null);
    });

    describe('if value `branch` is valid,', () => {
        it('extracts from `store` field description object', () => {
            const state = {
                remote: {
                    branch: {
                        fields: [{
                            groupName: 'Контактная информация',
                            hint: 'Как в паспорте.',
                            id: 'latin_lastname',
                            jsonSchema: {
                                maxLength: 100,
                                pattern: 'pattern',
                                type: 'string',
                            },
                            name: 'Фамилия',
                            type: 'input',
                        }],
                    },
                },
                store: {
                    latin_lastname: {
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$value: null,
                        $$visible: true,
                    },
                },
                local: {},
                router: {},
                simpleForm: {},
            };

            const type = 'input';

            expect(selectStoreFieldByFieldDefType(state, type)).toEqual({
                $$error: null,
                $$focus: false,
                $$hint: null,
                $$pending: false,
                $$touched: false,
                $$value: null,
                $$visible: true,
            });
        });

        it('returns `null` when value `type` does not coincides with one of the `fields`', () => {
            const state = {
                remote: {
                    branch: {
                        fields: [{
                            groupName: 'Контактная информация',
                            hint: 'Как в паспорте.',
                            id: 'latin_lastname',
                            jsonSchema: {
                                maxLength: 100,
                                pattern: 'pattern',
                                type: 'string',
                            },
                            name: 'Фамилия',
                            type: 'input',
                        }],
                    },
                },
                store: {
                    latin_lastname: {
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$value: null,
                        $$visible: true,
                    },
                },
                local: {},
                router: {},
                simpleForm: {},
            };

            const type = 'data';

            expect(selectStoreFieldByFieldDefType(state, type)).toBe(null);
        });

        it('returns `null` when value `type` does not coincides with one of the `store`', () => {
            const state = {
                remote: {
                    branch: {
                        fields: [{
                            groupName: 'Контактная информация',
                            hint: 'Как в паспорте.',
                            id: 'latin_lastname',
                            jsonSchema: {
                                maxLength: 100,
                                pattern: 'pattern',
                                type: 'string',
                            },
                            name: 'Фамилия',
                            type: 'input',
                        }],
                    },
                },
                store: {
                    email: {
                        $$error: null,
                        $$focus: false,
                        $$hint: null,
                        $$pending: false,
                        $$touched: false,
                        $$value: null,
                        $$visible: true,
                    },
                },
                local: {},
                router: {},
                simpleForm: {},
            };

            const type = 'input';

            expect(selectStoreFieldByFieldDefType(state, type)).toBe(null);
        });
    });
});

describe('selectFieldDefsIds()', () => {
    it('returns `null` when value `branch` is undefined', () => {
        const state = {
            store: {},
            remote: {},
            local: {},
            router: {},
            simpleForm: {},
        };

        expect(selectFieldDefsIds(state)).toEqual([]);
    });

    it('returns array of values `fields.id`', () => {
        const state = {
            store: {},
            remote: {
                branch: {
                    fields: [{
                        groupName: 'Контактная информация',
                        hint: 'Как в паспорте.',
                        id: 'latin_lastname',
                        jsonSchema: {
                            maxLength: 100,
                            pattern: 'pattern',
                            type: 'string',
                        },
                        name: 'Фамилия',
                        type: 'input',
                    }],
                },
            },
            local: {},
            router: {},
            simpleForm: {},
        };

        expect(selectFieldDefsIds(state)).toEqual(['latin_lastname']);
    });
});

describe('selectSubmitPayload()', () => {
    it('returns decamelize value of `serializedFields`, `countryId` and `branchId`', () => {
        const state = {
            local: {
                countryId: '225',
                branchId: 'russia_ur',
                serializedFields: {
                    cyrillic_last_name: 'Иванов',
                    cyrillic_first_name: 'Иван',
                    phone: '+79812345671',
                },
            },
            remote: {},
            router: {},
            store: {},
            simpleForm: {},
        };

        expect(selectSubmitPayload(state)).toEqual({
            country_id: '225',
            branch_id: 'russia_ur',
            fields: {
                cyrillic_last_name: 'Иванов',
                cyrillic_first_name: 'Иван',
                phone: '+79812345671',
            },
        });
    });

    it('returns decamelize value of `serializedFields`, `countryId` and `branchId` for project mobile mediation', () => {
        const state = {
            local: {
                countryId: '225',
                branchId: 'russia_ur',
                serializedFields: {
                    cyrillic_last_name: 'Иванов',
                    cyrillic_first_name: 'Иван',
                    phone: '+79812345671',
                },
            },
            remote: {},
            router: {
                location: {
                    pathname: '/form/mobile_mediation/',
                    search: 'test',
                    hash: 'test',
                    key: 'test',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectSubmitPayload(state)).toEqual({
            country_id: '225',
            branch_id: 'russia_ur',
            fields: {
                cyrillic_last_name: 'Иванов',
                cyrillic_first_name: 'Иван',
                phone: '+79812345671',
            },
            project: 'mobile_mediation',
        });
    });
});
