// @flow
'use strict';

import {
    OFFER_PAGE_ID,
    FORM_PAGE_ID,
    DEFAULT_PROJECT_ID,
    MOBILE_MEDIATION_PROJECT_ID,
    VIDEO_BLOGGER_PROJECT_ID,
    EFIR_BLOGGER_PROJECT_ID,
    GAMES_PROJECT_ID,
    ASSESSOR_PROJECT_ID,
    selectPageId,
    selectProjectId,
    selectSubpath,
} from '../../src/selectors';

describe('selectPageId()', () => {
    it('returns `null` when location is `undefined`', () => {
        const state = {
            local: {},
            remote: {},
            router: {},
            store: {},
            simpleForm: {},
        };

        expect(selectPageId(state)).toBe(null);
    });

    it('returns `OFFER_PAGE_ID` when location is set to `/form/offer`', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/form/offer/',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectPageId(state)).toBe(OFFER_PAGE_ID);
    });

    it('returns `FORM_PAGE_ID` when location is set to `/form`', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/form',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectPageId(state)).toBe(FORM_PAGE_ID);

        state.router.location.pathname = '/form/';
        expect(selectPageId(state)).toBe(FORM_PAGE_ID);

        state.router.location.pathname = '/form/mobile_mediation/';
        expect(selectPageId(state)).toBe(FORM_PAGE_ID);
    });
});

describe('selectSubpath()', () => {
    it('returns `null` when location is `undefined`', () => {
        const state = {
            local: {},
            remote: {},
            router: {},
            store: {},
            simpleForm: {},
        };

        expect(selectSubpath(state)).toBe(null);
    });

    it('returns `null` when location doesn\'t start from `BASE_URL`', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/some_path/to/somewhere',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectSubpath(state)).toBe(null);
    });

    it('returns `null` when location starts from `BASE_URL` but doesn\'t contain subpath', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/form/',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectSubpath(state)).toBe(null);
    });

    it('returns subpath when location starts from `BASE_URL` and contains subpath', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/form/offer',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectSubpath(state)).toBe('offer');

        state.router.location.pathname = '/form/mobile_mediation/';
        expect(selectSubpath(state)).toBe('mobile_mediation');

        state.router.location.pathname = '/form/multiple/subpaths/in/path';
        expect(selectSubpath(state)).toBe('multiple');
    });
});

describe('selectProjectId()', () => {
    it('returns `DEFAULT_PROJECT` when location is `undefined`', () => {
        const state = {
            local: {},
            remote: {},
            router: {},
            store: {},
            simpleForm: {},
        };

        expect(selectProjectId(state)).toBe(DEFAULT_PROJECT_ID);
    });

    it('returns `DEFAULT_PROJECT` when location doesn\'t start from `BASE_URL`', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/some_path/to/somewhere',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectProjectId(state)).toBe(DEFAULT_PROJECT_ID);
    });

    it('returns `DEFAULT_PROJECT` when location starts from `BASE_URL` but doesn\'t contain subpath', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/form/',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectProjectId(state)).toBe(DEFAULT_PROJECT_ID);
    });

    it('returns `MOBILE_MEDIATION_PROJECT` when location contains `mobile_mediation` subpath', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/form/mobile_mediation/',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectProjectId(state)).toBe(MOBILE_MEDIATION_PROJECT_ID);
    });

    it('returns `EFIR_BLOGGER_PROJECT` when location contains `efir_blogger` subpath', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/form/efir_blogger/',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectProjectId(state)).toBe(EFIR_BLOGGER_PROJECT_ID);
    });

    it('returns `VIDEO_BLOGGER_PROJECT` when location contains `video_blogger` subpath', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/form/video_blogger/',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectProjectId(state)).toBe(VIDEO_BLOGGER_PROJECT_ID);
    });

    it('returns `GAMES_PROJECT` when location contains `games` subpath', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/form/games/',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectProjectId(state)).toBe(GAMES_PROJECT_ID);
    });

    it('returns `ASSESSOR_PROJECT` when location contains `assessor` subpath', () => {
        const state = {
            local: {},
            remote: {},
            router: {
                location: {
                    hash: '',
                    key: '',
                    pathname: '/form/assessor/',
                    search: '',
                },
            },
            store: {},
            simpleForm: {},
        };

        expect(selectProjectId(state)).toBe(ASSESSOR_PROJECT_ID);
    });
});
