// @flow
'use strict';

import {
    selectLanguage,
    selectOfferId,
    selectFiasByGuid,
    selectUserData,
} from '../../src/selectors/remote.ts';

describe('selectLanguage()', () => {
    it('returns `null` when value `user` is undefined', () => {
        const state = {
            store: {},
            remote: {},
            local: {},
            router: {},
        };

        expect(selectLanguage(state)).toBe('ru');
    });

    it('returns value language', () => {
        const state = {
            store: {},
            remote: {
                user: {
                    canFillForm: true,
                    canonicalLogin: 'bes-test-024',
                    displayLogin: 'BeS.teST-024',
                    force: {
                        branchId: 'ag_ru',
                        countryId: '122',
                    },
                    hasRoleInPartner2: false,
                    isLogged: true,
                    userId: '607369863',
                    language: 'eng',
                },
            },
            local: {},
            router: {},
        };

        expect(selectLanguage(state)).toBe('eng');
    });
});

describe('selectOfferId()', () => {
    it('returns `null` when value `countryId` is undefined', () => {
        const state = {
            store: {},
            remote: {},
            local: {},
            router: {},
        };

        expect(selectOfferId(state)).toBe(null);
    });

    it('returns `null` when if value `branchId` is undefined', () => {
        const state = {
            remote: {},
            store: {},
            local: {
                countryId: '225',
            },
            router: {},
        };

        expect(selectOfferId(state)).toBe(null);
    });

    it('returns `null` when value `tree` is undefined', () => {
        const state = {
            remote: {},
            store: {},
            local: {
                countryId: '225',
                branchId: 'russia_ur',
            },
            router: {},
        };

        expect(selectOfferId(state)).toBe(null);
    });

    it('returns `null` when value `ofertaId` is empty string', () => {
        const state = {
            remote: {
                tree: {
                    '93': {
                        countryName: 'Аргентина',
                        branches: {
                            ag_ph: {
                                branchName: 'AAA',
                                ofertaId: '',
                                oferta: true,
                                ofertaUrl: 'ofertaUrl',
                            },
                            ag_ur: {
                                branchName: 'NNN',
                                ofertaId: 'oferta_ag_ur',
                                oferta: true,
                                ofertaUrl: 'ofertaUrl',
                            },
                        },
                        order: ['ag_ph', 'ag_ur'],
                    },
                },
            },
            store: {},
            local: {
                countryId: '93',
                branchId: 'ag_ph',
            },
            router: {},
        };

        expect(selectOfferId(state)).toBe(null);
    });

    it('returns value of `ofertaId`', () => {
        const state = {
            remote: {
                tree: {
                    '93': {
                        countryName: 'Аргентина',
                        branches: {
                            ag_ph: {
                                branchName: 'AAA',
                                ofertaId: 'oferta_ag_ph',
                                oferta: true,
                                ofertaUrl: 'ofertaUrl',
                            },
                            ag_ur: {
                                branchName: 'NNN',
                                ofertaId: 'oferta_ag_ur',
                                oferta: true,
                                ofertaUrl: 'ofertaUrl',
                            },
                        },
                        order: ['ag_ph', 'ag_ur'],
                    },
                },
            },
            store: {},
            local: {
                countryId: '93',
                branchId: 'ag_ph',
            },
            router: {},
        };

        expect(selectOfferId(state)).toBe('oferta_ag_ph');
    });
});

describe('selectFiasByGuid()', () => {
    it('returns `null` when value `fias` is undefined', () => {
        const state = {
            store: {},
            remote: {},
            local: {},
            router: {},
        };

        const guid = null;

        expect(selectFiasByGuid(state, guid)).toBe(null);
    });

    it('returns fias value of `ROOT` when value `guid` is null', () => {
        const state = {
            store: {},
            remote: {
                fias: {
                    ROOT: [{
                        formalName: 'Test',
                        guid: 'aea6280f-4648-460f-b8be-c2bc18923191',
                        objLevel: '1',
                        shortName: 'test',
                    }],
                    'a8cf3eb8-4689-4981-b908-9a760aae99d5': [{
                        formalName: 'Second test',
                        guid: 'ccdfd496-8108-4655-aadd-bd228747306d',
                        objLevel: '1',
                        shortName: 'second test',
                    }],
                },
            },
            local: {},
            router: {},
        };

        const guid = null;

        expect(selectFiasByGuid(state, guid)).toEqual([{
            formalName: 'Test',
            guid: 'aea6280f-4648-460f-b8be-c2bc18923191',
            objLevel: '1',
            shortName: 'test',
        }]);
    });

    it('returns value fias when value `guid` is valid', () => {
        const state = {
            store: {},
            remote: {
                fias: {
                    ROOT: [{
                        formalName: 'Test',
                        guid: 'aea6280f-4648-460f-b8be-c2bc18923191',
                        objLevel: '1',
                        shortName: 'test',
                    }],
                    'a8cf3eb8-4689-4981-b908-9a760aae99d5': [{
                        formalName: 'Second test',
                        guid: 'ccdfd496-8108-4655-aadd-bd228747306d',
                        objLevel: '1',
                        shortName: 'second test',
                    }],
                },
            },
            local: {},
            router: {},
        };

        const guid = 'a8cf3eb8-4689-4981-b908-9a760aae99d5';

        expect(selectFiasByGuid(state, guid)).toEqual([{
            formalName: 'Second test',
            guid: 'ccdfd496-8108-4655-aadd-bd228747306d',
            objLevel: '1',
            shortName: 'second test',
        }]);
    });
});

describe('selectUserData()', () => {
    it('returns `null` when value `user` is undefined', () => {
        const state = {
            store: {},
            remote: {},
            local: {},
            router: {},
        };

        expect(selectUserData(state)).toBe(null);
    });

    it('returns value of `user`', () => {
        const state = {
            store: {},
            remote: {
                user: {
                    canFillForm: true,
                    canonicalLogin: 'bes-test-024',
                    displayLogin: 'BeS.teST-024',
                    force: {
                        branchId: 'ag_ru',
                        countryId: '122',
                    },
                    hasRoleInPartner2: false,
                    isLogged: true,
                    userId: '607369863',
                    language: 'eng',
                },
            },
            local: {},
            router: {},
        };

        expect(selectUserData(state)).toEqual({
            canFillForm: true,
            canonicalLogin: 'bes-test-024',
            displayLogin: 'BeS.teST-024',
            force: {
                branchId: 'ag_ru',
                countryId: '122',
            },
            hasRoleInPartner2: false,
            isLogged: true,
            userId: '607369863',
            language: 'eng',
        });
    });
});
