'use strict';

const { camelCase, get, omit } = require('lodash');
const { mergeCallstack } = require('./utils/text');
const { resolve } = require('path');
const cfg = require('./config');
const debug = require('debug')('krush:access');
const express = require('express');
const bodyParser = require('body-parser');

const Github = require('./utils/Github');
const Startrack = require('./utils/Startrack');
const shared = require('./stories/shared');
const stories = require('./stories');

const app = express();

const github = new Github({token: cfg.githubToken});
const startrack = new Startrack({token: cfg.startrackToken});
const unwantedHeaders = [
  'accept',
  'accept-encoding',
  'accept-language',
  'cache-control',
  'connection',
  'cookie',
  'host',
  'pragma',
  'upgrade-insecure-requests',
];

app.use('/dist', express.static(resolve(__dirname, '../dist')));

app.use((req, res, next) => {
  const headers = omit(req.headers, unwantedHeaders);
  debug(`    → ${req.method} ${req.url} ${JSON.stringify(headers)}`);
  next();
});

app.get('/', (req, res) => res.status(200).end('hello, wanderer!'));

app.get('/secret/:story', (req, res) => {
  const story = shared[req.params.story];

  if (typeof story !== 'function') {
    return void res.status(404).end('Not found');
  }

  const context = req.query;

  (story.length === 3
    ? story(github, startrack, context)
    : story(github, context))
    .then(response => res.status(200).json(response))
    .catch(er => res.status(500).json({
      error: er.message,
      stack: mergeCallstack(er.stack),
    }));
});

app.get('/update', (req, res) => {
  const owner = req.query.owner;
  const repo = req.query.repo;

  if (!owner || !repo) {
    return void res.status(400).end('King Krush expects repo and owner params');
  }

  github.listPullRequests({owner: owner, repo: repo}, {state: 'open'})
    .then(prs => {
      const pending = Promise.all(prs
        .map(pr => shared.updateCodeReviewStatus(github, {
          author: get(pr, 'user.login'),
          number: pr.number,
          owner: owner,
          repo: repo,
        })));

      return pending;
    })
    .then(rs => res.status(200).json(rs))
    .catch(er => res.status(500).end(er.message));
});

app.use(bodyParser.json());
app.use('/webhooks', (req, res, next) => {
  const body = req.body;
  const delivery = req.headers['x-github-delivery'];
  const event = camelCase(req.headers['x-github-event']);

  if (stories.hasHook(event)) {
    return void stories(event, body, github, startrack)
      .then(response => res.status(201).json(response))
      .catch(er => {
        res.status(500).json({
          error: er.message,
          stack: mergeCallstack(er.stack),
        });

        next(er);
      });
  }

  res.status(202).json({
    delivery: delivery,
    event: event,
  });
});

app.use('*', (req, res) => res.status(404).end('Not found'));

app.use((err, req, res, next) => {
  console.error(err.stack || err);
  res.status(500).end();
});

module.exports = app;
