'use strict';

const extractIssueId = require('../utils/helper').extractIssueId;
const debug = require('debug')('krush:pullRequest');
const {
  includes,
  result,
  toLower,
} = require('lodash');
const {
  findReviewers,
  provideInformationAboutPullRequestToTicket,
  provideInformationAboutTicketToPullRequest,
  resetCodeReviewStatus,
  updateCodeReviewStatus,
  updateTicketStatusToMerged,
  getFreePort,
  getLastImageDB,
  autoCreateBeta,
  provideError,
  getProductionBackend,
} = require('./shared');
const cfg = require('../config');

/**
 * @param  {object} body.action                  'assigned', 'unassigned', 'labeled', 'unlabeled', 'opened', 'closed', or 'reopened', or 'synchronize'
 * @param  {string} body.pull_request.html_url
 * @param  {string} body.pull_request.head.ref   'branch'
 * @param  {string} body.number                  Номер issue / pr
 * @param  {string} body.repository.owner.login  owner / org
 * @param  {string} body.repository.name         repo
 * @param  {string} body.sender.login
 * @param  {string} body.pull_request.merged     true/false-flag
 * @param  {string} body.pull_request.body
 * @return {object}
 */
function extractContext(body) {
  return {
    action: body.action,
    author: result(body, 'sender.login'),
    branch: result(body, 'pull_request.head.ref'),
    number: body.number,
    owner: toLower(result(body, 'repository.owner.login')),
    repo: toLower(result(body, 'repository.name')),
    url: result(body, 'pull_request.html_url'),
    merged: result(body, 'pull_request.merged'),
    user: result(body, 'pull_request.user.login'),
    description: result(body, 'pull_request.body'),
  };
}

/**
 *
 * @param {string} description
 * @returns {object}
 */
function getParamsFromDescription(description) {
  const extract = (prefix) => {
    const expression = new RegExp(`${prefix}:\\s*\`?(.*\\w)\`?\\s*`);
    const match = description.match(expression);
    return match ? match[1] : null;
  };

  return {
    backend: extract('Backend'),
    blackbox: extract('Blackbox'),
    yacotools: extract('Yacotools'),
    bk: extract('BK'),
    typeDB: extract('TypeDB')
  };
}

/**
 * @param {object} context
 * @return {boolean}
 */
function shouldCreateBeta(context) {
  return context.repo === 'yharnam';
}

/**
 * @param {object} context
 * @return {promise}
 */
function createBeta(context) {
  const {backend, blackbox, yacotools, bk, typeDB} = getParamsFromDescription(context.description);
  debug(`createBeta backend=${backend}, blackbox=${blackbox}, yacotools=${yacotools}, bk=${bk}, typeDB=${typeDB}`);

  if (!backend || !blackbox || !yacotools || !bk || !typeDB) {
    debug(`createBeta, insufficient params`);
    return null;
  }

  const getPort = getFreePort();
  const getImageDB = (typeDB === 'docker') ? getLastImageDB() : null;
  const getBackend = (backend === 'production') ? getProductionBackend() : backend;

  return Promise.all([getPort, getImageDB, getBackend])
    .then(
      ([port, imageDB, backendBranch]) => {
        debug(`Begin create beta: port - ${port}, typeDB - ${typeDB}, imageDB - ${imageDB}, frontend - ${context.branch}, backend - ${backendBranch}`);
        const options = {
          port,
          imageDB,
          blackbox,
          yacotools,
          bk,
          typeDB,
          backend: backendBranch,
          frontend: context.branch,
          comment: `Для ревью ${extractIssueId(context.branch)}`,
        };
        return autoCreateBeta(options);
      },
      er => {
        throw new Error(`Не удалось получить порт или образ БД:\n${er.code} - ${er.message}`);
      })
    .catch(
      er => {
        throw new Error(`Не удалось запустить создание беты:\n${er.code} - ${er.message}`);
      });
}

/**
 * @param {object} context
 * @param {Github} gh
 * @param {Startrack} st
 * @return {promise}
 */
function openedPullRequest(context, gh, st) {
  const isReleaseBranch = /^release\/\d+$/.test(context.branch) || /^frontend_release_[.\d]+$/.test(context.branch);

  debug(`openedPullRequest, repo=${context.repo}, branch=${context.branch} ${isReleaseBranch ? ', release' : ''}`);

  if (!isReleaseBranch && context.repo !== 'backend') { // DI
    debug(`openedPullRequest, repo=${context.repo}, branch=${context.branch}`);
    return findReviewers(st, context)
      .then(reviewers => {
        const stories = [];

        debug(`openPullRequests found reviewers: #${reviewers.length}`);
        if (reviewers.length > 0) {
          debug(`${reviewers} was assigned for ${context.url}`);

          stories.push(gh.addReviewersToAPullRequest(context, reviewers));
          stories.push(gh.addAssigneesToAnIssue(context, reviewers));
        }

        if (shouldCreateBeta(context)) {
          debug(`openedPullRequest, should create beta`);
          stories.push(createBeta(context, gh, st)
            .then(optsBeta => Promise.all([
                provideInformationAboutTicketToPullRequest(gh, st, context, optsBeta),
                provideInformationAboutPullRequestToTicket(null, st, context, reviewers, optsBeta),
              ])
            )
            .catch(er => provideError(gh, st, context, er))
          );
        }

        return stories;
      });
  }

  return Promise.resolve(true);
}

/**
 * @param  {object}    body
 * @param  {object}    body.action                  'assigned', 'unassigned', 'labeled', 'unlabeled', 'opened', 'closed', or 'reopened', or 'synchronize'
 * @param  {string}    body.pull_request.html_url
 * @param  {string}    body.pull_request.head.label 'partner:branch'
 * @param  {string}    body.pull_request.head.ref   'branch'
 * @param  {string}    body.number                  Номер issue / pr
 * @param  {string}    body.repository.owner.login  owner / org
 * @param  {string}    body.repository.name         repo
 * @param  {string}    body.sender.login
 * @param  {string}    body.pull_request.merged     true/false-flag
 * @param  {string}    body.pull_request.user.login     PR-owner
 * @param  {github}    gh
 * @param  {startrack} st
 * @return {promise}
 */
module.exports = function pullRequest(body, gh, st) {
  const context = extractContext(body);
  let stories = [];

  debug(`pullRequest action=${context.action}`);

  if (includes(['opened', 'reopened'], context.action)) {
    stories.push(resetCodeReviewStatus(gh, context));
    stories.push(openedPullRequest(context, gh, st));
  }

  if (context.action === 'synchronize') {
    stories.push(updateCodeReviewStatus(gh, context));
  }

  if (context.repo === 'partner2'
    && includes(cfg.resolve_tickets_for_users, context.user)
    && context.action === 'closed'
    && context.merged) {
    stories.push(updateTicketStatusToMerged(null, st, context));
  }

  if (stories.length === 0) {
    return Promise.resolve(body);
  }

  return Promise.all(stories);
};

module.exports.extractContext = extractContext;
