'use strict';

const {
  assign,
  chain,
  difference,
  identity,
  isEmpty,
  map,
  pick,
  result,
  spread,
  without,
} = require('lodash');
const cfg = require('../config');
const API = require('../utils/API');
const Agent = require('https').Agent;
const createCodeReviewStatus = require('../utils/status').createCodeReviewStatus;
const createCommitContext = require('../utils/helper').createCommitContext;
const estimateVotes = require('../utils/helper').estimateVotes;
const extractIssueId = require('../utils/helper').extractIssueId;
const filterCommentsByAuthor = require('../utils/helper').filterCommentsByAuthor;
const findLastCommit = require('../utils/helper').findLastCommit;
const findTeam = require('../utils/helper').findTeam;
const findTeammates = require('../utils/helper').findTeammates;
const listVotes = require('../utils/helper').listVotes;
const randomize = require('../utils/randomize');
const debug = require('debug')('krush:shared');

const ESTIMATION = require('../config').estimation;
const STATES = require('../utils/status').STATES;
const TEAMS = require('../config').teams;
const VACATION = require('../config').vacation;

const MIN_ESTIMATION = 2;

const agent = new Agent({
    keepAlive: true,
});

/**
 * @param  {startrack} st
 * @param  {object} context
 * @param  {string} context.author
 * @param  {string} context.number
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @return {promise}
 */
function findReviewers(st, context) {
  const issueId = extractIssueId(context.branch);

  debug(`findReviewer ${st}`);

  return st.getIssue(issueId)
    .then(
      issue => result(issue, 'reviewers', []).map(reviewer => reviewer.id)
    )
    .then(reviewers => {
      if (reviewers.length > 0) {
        return reviewers;
      }

      const reviewer = chooseReviewer(context);

      if (reviewer !== null) {
        return [reviewer];
      }

      return [];
    });
}

/**
 * @param  {object} context
 * @param  {string} context.author
 * @param  {string} context.number
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @return {promise}
 */
function chooseReviewer(context) {
  const participants = findTeammates(cfg.teams, cfg.repo, context.repo, context.owner, context.author);

  if (isEmpty(participants)) {
    return null;
  }

  const rndContext = context.author + '|' + context.repo;
  const availableParticipants = difference(participants, VACATION).sort();

  if (!context.author) {
    debug(`empty author, context: ${context}`);
  }
  const reviewer = randomize(rndContext, availableParticipants, context.author);

  debug(`Reviewer for repos ${context.repo} and author ${context.author} is ${reviewer}`);

  return reviewer;
}

/**
 * @param  {string} branch
 * @param  {string} param
 * @return {string}
 */
function getCreateBetaLink(branch, param = 'frontendBranch') {
  return `<a href='https://creator.partner.yandex-team.ru/create?${param}=${branch}'><img alt='Создать бету' width='122' height='32' src='https://krush.partner.yandex-team.ru/dist/create-beta-button.png'/></a>`;
}

/**
 *
 * @param  {object}     optsBeta
 * @param  {string}     optsBeta.backend
 * @param  {number}     optsBeta.port
 * @returns {string}
 */
function getMessageAboutBeta(optsBeta) {
  return `\n\nЗапущена сборка беты.\nBackend: ${optsBeta.backend}\n` +
    `Ссылка: https://partner-creator01f.yandex.ru:${optsBeta.port}\n` +
    `Creator: https://creator.partner.yandex-team.ru/beta/myt/${optsBeta.port}`;
}

/**
 * @param  {*}         _
 * @param  {startrack} st
 * @param  {object}    context
 * @param  {string}    context.author
 * @param  {string}    context.branch
 * @param  {string}    context.repo
 * @param  {string}    context.url
 * @param  {string}    reviewers
 * @param  {?object}   optsBeta
 * @param  {string}    optsBeta.backend
 * @param  {number}    optsBeta.port
 * @return {promise}
 */
function provideInformationAboutPullRequestToTicket(_, st, context, reviewers = [], optsBeta = null) {
  const issueId = extractIssueId(context.branch);

  debug(`provideInformationAboutPullRequestToTicket ${issueId} ${optsBeta ? 'with' : 'without'} beta options`);

  if (!issueId) {
    return Promise.resolve(context);
  }

  const msg = {
    summonees: [],
    text: `%%${context.repo}: ${context.branch}%%\nPR: ${context.url}`,
  };

  if (reviewers.length > 0) {
    msg.summonees = msg.summonees.concat(reviewers);

    if (reviewers.length == 1) {
      msg.text = `staff:${reviewers}, сделай ревью, пожалуйста.\n\n${msg.text}`;
    } else {
      const staffReviewers = reviewers.map(reviewer => `staff:${reviewer}`).join(', ');
      msg.text = `${staffReviewers}, проведите ревью, пожалуйста.\n\n${msg.text}`;
    }
  }

  if (optsBeta) msg.text += getMessageAboutBeta(optsBeta);

  if (['partner2', 'yharnam'].includes(context.repo)) {
    const param = context.repo === 'yharnam' ? 'frontendBranch' : 'backendBranch';

    msg.text += '\n\n<# ' + getCreateBetaLink(context.branch, param) + ' #>';
  }

  return st.createAComment(issueId, msg);
}

/**
 * @param  {github}    gh
 * @param  {startrack} st
 * @param  {object}    context
 * @param  {string}    context.branch
 * @param  {string}    context.number
 * @param  {string}    context.owner
 * @param  {string}    context.repo
 * @param  {?object}   optsBeta
 * @param  {string}    optsBeta.backend
 * @param  {number}    optsBeta.port
 * @return {promise}
 */
function provideInformationAboutTicketToPullRequest(gh, st, context, optsBeta) {
  const issueId = extractIssueId(context.branch);

  debug(`provideInformationAboutTicketToPullRequest ${issueId} ${optsBeta ? 'with' : 'without'} beta options`);

  if (!issueId) {
    return Promise.resolve(context);
  }

  return st.getIssue(issueId)
    .then(issue => {
      let msg = `# [${issueId}](https://st.yandex-team.ru/${issueId})`;

      const author = result(issue, 'createdBy.display');
      const summary = result(issue, 'summary');

      if (author) {
        msg += `\n> ${summary}\n\n*— ${author}*`;
      }

      if (optsBeta) msg += getMessageAboutBeta(optsBeta);

      if (['partner2', 'yharnam'].includes(context.repo)) {
        const param = context.repo === 'yharnam' ? 'frontendBranch' : 'backendBranch';

        msg += '\n\n' + getCreateBetaLink(context.branch, param);
      }

      return gh.createAComment(context, msg);
    });
}

/**
 *
 * @param   {github}    gh
 * @param   {startrack} st
 * @param   {object}    context
 * @param   {string}    error
 * @returns {promise}
 */
function provideError(gh, st, context, error) {
  const issueId = extractIssueId(context.branch);
  const text = `Произошла ошибка.\n${error}`;
  const stories = [];

  if (!issueId) {
    return Promise.resolve(context);
  }
  if (gh) stories.push(gh.createAComment(context, text));
  if (st) stories.push(st.createAComment(issueId, {text}));

  return Promise.all(stories);
}

/**
 * @param  {github} gh
 * @param  {object} context
 * @param  {string} context.number
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @return {promise}
 */
function resetCodeReviewStatus(gh, context) {
  debug(`resetCodeReviewStatus ${context.number} ${context.owner} ${context.repo}`);
  const status = createCodeReviewStatus(STATES.PENDING, 'Нужно два положительных голоса для успеха');

  return gh.listLastCommitsOnAPullRequest(context)
    .then(findLastCommit)
    .then(lastCommit => createCommitContext(lastCommit, context))
    .then(commitContext => gh.createAStatus(commitContext, status));
}

/**
 * @param  {github} gh
 * @param  {object} context
 * @param  {string} context.author
 * @param  {string} context.number
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {string} context.reviewer?
 * @param  {string} context.url
 * @return {promise}
 */
function updateCodeReviewStatus(gh, context) {
  return Promise.all([
    gh.listCommentsOnAnIssue(context, {query: {per_page: 100}})
      .then(comments => filterCommentsByAuthor(comments, context.author))
      .then(listVotes),
    gh.listLastCommitsOnAPullRequest(context)
      .then(findLastCommit)
      .then(lastCommit => createCommitContext(lastCommit, context)),
    gh.listLabelsOnAnIssue(context)
      .then(labels => map(labels, 'name')),
  ])
    .then(spread((votes, commitContext) => {
      const author = context.reviewer || context.author;
      const estimation = estimateVotes(votes, [cfg.servant]);
      const members = map(votes.filter(vote => vote.estimation > 0), 'reviewer');
      const team = findTeam(TEAMS, author);
      const minEstimation = ESTIMATION[team] || MIN_ESTIMATION;
      const status = createCodeReviewStatus(estimation >= minEstimation
          ? STATES.SUCCESS
          : STATES.PENDING,
        `Результат голосования: ${estimation}`);

      const labelsContext = assign(pick(context, ['number', 'owner', 'repo']), {members: members});

      debug(`${author} from team ${JSON.stringify(team)} voted, result ${estimation}/${minEstimation} (${JSON.stringify(members)}) for pull ${context.url}`);

      return Promise.all([
        gh.createAStatus(commitContext, status),
        updateLabels(gh, labelsContext),
      ]);
    }));
}

/**
 * @param  {github} gh
 * @param  {object} context
 * @param  {string} context.number
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {array}  context.members
 * @param  {array}  [context.protected]
 * @return {promise}
 */
function updateLabels(gh, context) {
  const members = identity(context.members);

  return Promise.all([
    gh.listLabelsOnAnIssue(context)
      .then(labels => map(labels, 'name')),
    gh.listAllLabelsForThisRepository(context)
      .then(labels => map(labels, 'name')),
  ])
    .then(spread((labelsOnIssue, labelsOnPR) => {
      const labelsList = chain(labelsOnIssue).intersection(context.protected).concat(members).uniq().value();
      const labelsToBeCreated = without.apply(null, [labelsList].concat(labelsOnPR));

      return Promise.all([labelsToBeCreated.map(name => gh.createALabel(context, {name: name, color: 'f6f5f3'}))])
        .then(_ => gh.addLabelsToAnIssue(context, labelsList));
    }));
}

/**
 * @param  {*}         _
 * @param  {startrack} st
 * @param  {object}    context
 * @param  {string}    context.branch
 * @return {promise}
 */
function updateTicketStatusToMerged(_, st, context) {
  const issueId = extractIssueId(context.branch);

  if (!issueId) {
    return Promise.resolve(context);
  }

  return st.mergeIssue(issueId);
}

/**
 *
 * @returns {promise}
 */
function getFreePort() {
    const opts = {
        agent: agent,
        uri: 'https://creator-myt.partner.yandex-team.ru/',
        token: '',
    };
    const api = new API(opts);

    return api.get('/api/3/betas/', {rejectUnauthorized: false})
        .then(betas => betas.filter(beta => beta && beta.status === 'free'))
        .then(betas => {
          if (!betas.length) {
            throw new Error('Free beta port not found');
          }

          debug(`getFreePort found one: ${betas[0].port}`);
          return Number(betas[0].port);
        });
}

function getLastImageDB() {
    const opts = {
        agent: agent,
        uri: 'https://creator-myt.partner.yandex-team.ru/',
        token: '',
    };
    const api = new API(opts);

    return api.get('/api/3/cached_docker_db_images', {rejectUnauthorized: false})
        .then(images => images.reverse()[0])
}

function getProductionBackend() {
    const opts = {
      agent: agent,
      uri: 'https://partner.yandex-team.ru/',
      token: '',
    };
    const api = new API(opts);

    return api.get('/partner2_production_version.json', {rejectUnauthorized: false})
      .then(result => result.s)
}

/**
 *
 * @param {object}    options
 * @param {number}    options.port
 * @param {string}    options.imageDB
 * @param {string}    options.frontend
 * @param {string}    options.backend
 * @param {string}    options.blackbox
 * @param {string}    options.yacotools
 * @param {string}    options.bk
 * @param {string}    options.comment
 * @param {number}    options.ttl
 * @return {promise}
 */
function autoCreateBeta(options) {
    const {port, imageDB, frontend = 'master', backend = 'master', blackbox, yacotools, bk, comment = 'Для ревью', ttl = 172800, typeDB = 'docker'} = options;

    debug(`autoCreateBeta port=${port} imageDB=${imageDB} frontend=${frontend}, ...`);
    const db = {};

    if (typeDB === 'docker') {
      Object.assign(db, {
        type: 'docker',
        docker_image: imageDB,
      });
    } else {
      Object.assign(db, {
        type: 'preset',
        preset_id: typeDB
      });
    }

    const body = {
        backend,
        comment,
        frontend,
        ttl,
        db,
        blackbox: {
            type: 'url',
            url: blackbox,
        },
        bs: {
            type: 'url',
            url: bk,
        },
        yacotools: {
            type: 'url',
            url: yacotools,
        }
    };
    const opts = {
        agent: agent,
        uri: 'https://creator-myt.partner.yandex-team.ru/',
        token: '',
    };
    const userOpts = {
        betaPort: port,
        rejectUnauthorized: false,
    };
    const api = new API(opts);

    return api.put(`/api/3/betas/${port}`, body, userOpts)
        .then(() => {
            debug('autoCreateBeta, success');
            return {
                backend,
                port
            }
        });
}

exports.findReviewers = findReviewers;
exports.chooseReviewer = chooseReviewer;
exports.getCreateBetaLink = getCreateBetaLink;
exports.provideInformationAboutPullRequestToTicket = provideInformationAboutPullRequestToTicket;
exports.provideInformationAboutTicketToPullRequest = provideInformationAboutTicketToPullRequest;
exports.resetCodeReviewStatus = resetCodeReviewStatus;
exports.updateCodeReviewStatus = updateCodeReviewStatus;
exports.updateLabels = updateLabels;
exports.updateTicketStatusToMerged = updateTicketStatusToMerged;
exports.getFreePort = getFreePort;
exports.getLastImageDB = getLastImageDB;
exports.autoCreateBeta = autoCreateBeta;
exports.provideError = provideError;
exports.getProductionBackend = getProductionBackend;
