'use strict';

const cfg = require('../config');
const chain = require('lodash').chain;
const extractIssueId = require('../utils/helper').extractIssueId;
const includes = require('lodash').includes;
const map = require('lodash').map;
const result = require('lodash').result;
const spread = require('lodash').spread;

/**
 * @param  {object} body
 * @return {object}
 */
function extractContext(body) {
  return {
    branches: map(body.branches, 'name'),
    owner: result(body, 'repository.owner.login'),
    repo: result(body, 'repository.name'),
  };
}

/**
 * @param  {object}   body
 * @param  {string}   body.commit.sha
 * @param  {object[]} body.branches -> {name: '', commit: {}}
 * @param  {string}   body.repository.name
 * @param  {string}   body.repository.owner.login
 * @param  {object}   gh
 * @param  {object}   st
 */
module.exports = function status(body, gh, st) {
  const context = extractContext(body);

  return gh.listPullRequests(context, {state: 'open'})
    .then(prs => chain(prs)
      .map(pr => ({
        author: pr.user.login,
        number: pr.number,
        name: result(pr, 'head.ref'),
        owner: context.owner,
        repo: context.repo,
        sha: result(pr, 'head.sha'),
        ticket: extractIssueId(result(pr, 'head.ref')),
      }))
      .filter(pr => includes(context.branches, pr.name))
      .filter(pr => Boolean(pr.ticket))
      .value())
    .then(prs => Promise.all([
      Promise.all(prs.map(pr => st.getIssue(pr.ticket))),
      Promise.all(prs.map(pr => gh.getTheCombinedStatus(pr))),
      prs,
    ]))
    .then(spread((tickets, statuses, prs) => tickets
      .map((ticket, i) => {
        if (result(statuses[i], 'state') !== 'success') {
          return ticket;
        }

        if (result(ticket, 'status.key') === 'codeReview' && includes(cfg.resolve_tickets_for_users, prs[i].author)) {
          return st.resolveIssue(prs[i].ticket);
        }

        return ticket;
      })));
};

module.exports.extractContext = extractContext;
