'use strict';

const debug = require('debug')('krush:api');
const {defaults, merge} = require('lodash');
const got = require('got');
const url = require('url');

/**
 * @param  {object} opts
 * @param  {object} opts.agent
 * @param  {string} opts.uri
 * @param  {string} opts.token
 * @return {api}
 */
function API(opts) {
  opts = opts || {};

  this._agent = opts.agent;
  this._uri = opts.uri;
  this._token = opts.token;
}

/**
 * @param  {string} uri
 * @param  {object} [userOpts]
 * @return {promise}
 */
API.prototype.get = function (uri, userOpts) {
  return this.getRaw(uri, userOpts).then(resp => resp.body);
};

/**
 * @param  {string} uri
 * @param  {object} [userOpts]
 * @return {promise}
 */
API.prototype.getRaw = function (uri, userOpts) {
  return this._requestRaw('GET', uri, null, userOpts);
};

/**
 * @private
 * @param {string} method
 * @param {string} uri
 * @param {?(string|object)} body
 * @param {object} [userOpts]
 * @returns {promise}
 */
API.prototype._requestRaw = function (method, uri, body, userOpts) {
  const destination = this._buildUrl(uri);
  const opts = this._buildOpts(Object.assign({body}, userOpts));

  debug(`    → ${method} ${destination}`);
  const gotMeth = method.toLowerCase();

  return got[gotMeth](destination, opts).then(response => {
    debug(`${response.statusCode} ← ${method} ${destination}`);
    return response;
  }).catch(er => {
    debug(`${er.message}`);
    if (er.response) {
      const {response = {}} = er;
      const {body = null, statusMessage, statusCode} = response;
      debug(`${statusCode} ← ${method} ${destination} - ${er.message}`);
      const statusPhrase = statusMessage || (typeof body === 'string' ? body : '');
      const status = [String(statusCode), statusPhrase].filter(Boolean).join(' ');
      er.message = `${status} ← ${method} ${destination}`;
    } else {
      debug(`? ← ${method} ${destination} - ${er.message} (${String(er)})`);
    }
    throw er;
  });
};

API.prototype.delete = function (uri, userOpts) {
  return this._requestRaw('DELETE', uri, null, userOpts).then(resp => resp.body);
};

/**
 * @param  {string} uri
 * @param  {object} body
 * @param  {object} [userOpts]
 * @return {promise}
 */
API.prototype.patch = function (uri, body, userOpts) {
  return this._requestRaw('PATCH', uri, body, userOpts).then(resp => resp.body);
};

/**
 * @param  {string} uri
 * @param  {object} body
 * @param  {object} [userOpts]
 * @return {promise}
 */
API.prototype.post = function (uri, body, userOpts) {
  return this.postRaw(uri, body, userOpts).then(resp => resp.body);
};

/**
 * @param  {string} uri
 * @param  {object} body
 * @param  {object} [userOpts]
 * @return {promise}
 */
API.prototype.postRaw = function (uri, body, userOpts) {
  return this._requestRaw('POST', uri, body, userOpts);
};

/**
 * @param  {string} uri
 * @param  {object} body
 * @param  {object} [userOpts]
 * @return {promise}
 */
API.prototype.put = function(uri, body, userOpts) {
  return this._requestRaw('PUT', uri, body, userOpts).then(resp => resp.body);
};

/**
 * @private
 * @return {string}
 */
API.prototype._buildAuthorizationHeader = function () {
  return 'token ' + this._token;
};

/**
 * @private
 * @param  {object} [userOpts]
 * @return {object}
 */
API.prototype._buildOpts = function (userOpts) {
  const opts = merge({
    agent: this._agent,
    json: true,
    headers: {
      'user-agent': 'King Krush knocks to your door',
      'x-request-id': userOpts.requestId,
    },
  }, userOpts);

  if (typeof this._token === 'string') {
    opts.headers.authorization = this._buildAuthorizationHeader();
  }

  return opts;
};

/**
 * @private
 * @param {string} url
 * @return {string}
 */
API.prototype._buildUrl = function (uri) {
  const urlObj = url.parse(uri);

  if (typeof this._uri !== 'string' || urlObj.hostname) {
    return uri;
  }

  const baseObj = url.parse(this._uri);
  const final = defaults(baseObj, urlObj);

  final.pathname = [baseObj.pathname, urlObj.pathname]
    .filter(Boolean).join('/').replace(/\/+/g, '\/');
  final.search = urlObj.search;

  return url.format(final);
};

module.exports = API;
