'use strict';

const {Agent} = require('https');
const API = require('./API');
const {defaults, partial, result} = require('lodash');
const {findLastPage} = require('./helper');
const {inherits} = require('util');

const debug = require('debug')('krush:utils:Github');

const agent = new Agent({
  keepAlive: true,
});

/**
 * Events: https://developer.github.com/enterprise/2.4/webhooks/#events
 *
 * @class {Github}
 * @param {object} opts
 * @param {object} opts.agent
 * @param {string} opts.uri
 * @param {string} opts.token
 */
function Github(opts) {
  opts = opts || {};
  defaults(opts, {
    agent: agent,
    uri: 'https://github.yandex-team.ru/api/v3/'
  });

  API.call(this, opts);
}

inherits(Github, API);

/**
 * @param  {object} pull
 * @param  {string} pull.number
 * @param  {string} pull.owner
 * @param  {string} pull.repo
 * @param  {string} reviewers
 */
Github.prototype.addReviewersToAPullRequest = function (pull, reviewers) {
  // https://developer.github.com/v3/pulls/review_requests/#create-a-review-request
  return this.post(`/repos/${pull.owner}/${pull.repo}/pulls/${pull.number}/requested_reviewers`, {
    reviewers: reviewers,
  });
};

/**
 * @param  {object} issue
 * @param  {string} issue.number
 * @param  {string} issue.owner
 * @param  {string} issue.repo
 * @param  {string} assignee
 */
Github.prototype.addAssigneesToAnIssue = function (issue, assignees) {
  // https://developer.github.com/v3/issues/assignees/#add-assignees-to-an-issue
  return this.post(`/repos/${issue.owner}/${issue.repo}/issues/${issue.number}/assignees`, {
    assignees: assignees,
  });
};

/**
 * @param  {object} issue
 * @param  {string} issue.owner
 * @param  {string} issue.repo
 * @param  {array}  json
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.addLabelsToAnIssue = function (issue, json, userOpts) {
  // https://developer.github.com/v3/issues/labels/#add-labels-to-an-issue
  return this.post(`/repos/${issue.owner}/${issue.repo}/issues/${issue.number}/labels`, json, userOpts);
};

/**
 * @param  {object} issue
 * @param  {string} issue.number
 * @param  {string} issue.owner
 * @param  {string} issue.repo
 * @param  {string} body
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.createAComment = function (issue, body, userOpts) {
  // https://developer.github.com/v3/issues/comments/#create-a-comment
  return this.post(`/repos/${issue.owner}/${issue.repo}/issues/${issue.number}/comments`, {body: body}, userOpts);
};

/**
 * @param  {object} issue
 * @param  {string} issue.owner
 * @param  {string} issue.repo
 * @param  {object} json
 * @param  {string} json.name
 * @param  {string} json.color
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.createALabel = function (issue, json, userOpts) {
  // https://developer.github.com/v3/issues/labels/#create-a-label
  return this.post(`/repos/${issue.owner}/${issue.repo}/labels`, json, userOpts);
};

/**
 * @param  {object} commit
 * @param  {string} commit.owner
 * @param  {string} commit.repo
 * @param  {string} commit.sha
 * @param  {object} json
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.createAStatus = function (commit, json, userOpts) {
  // https://developer.github.com/v3/repos/statuses/#create-a-status
  return this.post(`/repos/${commit.owner}/${commit.repo}/statuses/${commit.sha}`, json, userOpts);
};

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {string} context.sha
 * @return {object}
 */
Github.prototype.getTheCombinedStatus = function (context) {
  // https://developer.github.com/v3/repos/statuses/#get-the-combined-status-for-a-specific-ref
  return this.get(`/repos/${context.owner}/${context.repo}/commits/${context.sha}/status`);
};

/**
 * @param  {object} issue
 * @param  {string} issue.number
 * @param  {string} issue.owner
 * @param  {string} issue.repo
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.listCommentsOnAnIssue = function (issue, userOpts) {
  // https://developer.github.com/v3/issues/comments/#list-comments-on-an-issue
  return this.get(`/repos/${issue.owner}/${issue.repo}/issues/${issue.number}/comments`, userOpts);
};

/**
 * @param  {object} pullRequest
 * @param  {string} pullRequest.number
 * @param  {string} pullRequest.owner
 * @param  {string} pullRequest.repo
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.listCommitsOnAPullRequest = function (pullRequest, userOpts) {
  // https://developer.github.com/v3/pulls/#list-commits-on-a-pull-request
  return this.get(`/repos/${pullRequest.owner}/${pullRequest.repo}/pulls/${pullRequest.number}/commits`, userOpts);
};

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.listAllLabelsForThisRepository = function (context, userOpts) {
  // https://developer.github.com/v3/issues/labels/#list-all-labels-for-this-repository
  return this.get(`/repos/${context.owner}/${context.repo}/labels`, userOpts);
};

/**
 * @param  {object} issue
 * @param  {string} issue.number
 * @param  {string} issue.owner
 * @param  {string} issue.repo
 * @return {promise}
 */
Github.prototype.listLabelsOnAnIssue = function (issue, userOpts) {
  // https://developer.github.com/v3/issues/labels/#list-labels-on-an-issue
  return this.get(`/repos/${issue.owner}/${issue.repo}/issues/${issue.number}/labels`, userOpts);
};

/**
 * @param  {object} pullRequest
 * @param  {string} pullRequest.number
 * @param  {string} pullRequest.owner
 * @param  {string} pullRequest.repo
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.listLastCommitsOnAPullRequest = function (pullRequest, userOpts) {
  debug(`listLastCommitsOnAPullRequest ${pullRequest.owner}/${pullRequest.repo} ${pullRequest.number}`);
  // https://developer.github.com/v3/pulls/#list-commits-on-a-pull-request
  return this.getRaw(`/repos/${pullRequest.owner}/${pullRequest.repo}/pulls/${pullRequest.number}/commits`, userOpts)
    .then(response => {
      const pagination = result(response, 'headers.link');
      const lastPage = findLastPage(pagination);

      debug(`listLastCommitsOnAPullRequest ... got ${lastPage ? 'page' : 'nothing'}`);

      if (lastPage) {
        return this.get(lastPage);
      }

      return response.body;
    });
};

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.listPullRequests = function (context, userOpts) {
  debug(`listPullRequests ${context.owner}/${context.repo}`);
  // https://developer.github.com/v3/pulls/#list-pull-requests
  return this.get(`/repos/${context.owner}/${context.repo}/pulls`, userOpts);
};

/**
 * @param  {string} teamId
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.listTeamMembers = function (teamId, userOpts) {
  debug(`listTeamMembers ${teamId}`);
  // https://developer.github.com/v3/orgs/teams/#list-team-members
  return this.get(`/teams/${teamId}/members`, userOpts);
};

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.listTeams = function (context, userOpts) {
  debug(`listTeams ${context.owner}/${context.repo}`);
  return this.get(`/repos/${context.owner}/${context.repo}/teams`, userOpts);
};

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {string} context.ref
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.getCommitForRef = function (context, userOpts) {
  debug(`getCommitForRef ${context.owner}/${context.repo} ${context.ref}`);
  // https://developer.github.com/enterprise/2.11/v3/repos/commits/#get-a-single-commit
  return this.get(`/repos/${context.owner}/${context.repo}/commits/${context.ref}`, userOpts);
};

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.listAllTags = function (context, userOpts) {
  debug(`listAllTags ${context.owner}/${context.repo}`);
  // https://developer.github.com/v3/git/refs/#get-all-references
  return this.get(`/repos/${context.owner}/${context.repo}/git/refs/tags`, userOpts);
};

Github.prototype.listAllBranches = function (context, userOpts) {
  debug(`listAllBranches ${context.owner}/${context.repo}`);
  // https://developer.github.com/v3/git/refs/#get-all-references
  return this.get(`/repos/${context.owner}/${context.repo}/git/refs/heads`, userOpts);
}

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {string} context.ref
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.listMatchingRefs = function (context, userOpts) {
  // https://developer.github.com/v3/git/refs/#list-matching-references
  const {owner, repo, ref} = context;
  debug(`listMatchingRefs ${owner}/${repo} ~ ${ref}`);
  const url = `/repos/${owner}/${repo}/git/matching-refs/${ref}`;
  return this.get(url, userOpts);
}

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {string} context.tag
 * @param  {string} context.sha
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.createATag = function (context, userOpts) {
  // https://developer.github.com/v3/git/refs/#create-a-reference

  const {sha, tag} = context;

  debug(`createATag: ${tag} -> ${sha}`);

  return this.createARef(context, {
    ref: `refs/tags/${tag}`,
    sha,
  }, userOpts);
};

/**
 * @param {string} type
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {string} context.sha
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.fetchObject = function (type, context, userOpts) {
  const {owner, repo, sha} = context;
  return this.get(`/repos/${owner}/${repo}/git/${type}/${sha}`, userOpts);
};

/**
 * @param {string} type
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param {object} payload
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.createObject = function (type, context, payload, userOpts) {
  const {owner, repo} = context;
  debug(`createObject ${context.owner}/${context.repo} ${type}`);
  return this.post(`/repos/${owner}/${repo}/git/${type}`, payload, userOpts);
};

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {string} context.sha
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.getCommit = partial(Github.prototype.fetchObject, 'commits');
Github.prototype.getTag = partial(Github.prototype.fetchObject, 'tags');
Github.prototype.getTree = partial(Github.prototype.fetchObject, 'trees');
Github.prototype.getBlob = partial(Github.prototype.fetchObject, 'blobs');

Github.prototype.createABlob = partial(Github.prototype.createObject, 'blobs');
Github.prototype.createACommit = partial(Github.prototype.createObject, 'commits');
Github.prototype.createATree = partial(Github.prototype.createObject, 'trees');
Github.prototype.createARef = partial(Github.prototype.createObject, 'refs');

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {string} context.sha
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.getRef = function (context, userOpts) {
  // https://developer.github.com/v3/git/refs/#get-a-single-reference
  const {owner, repo, ref} = context;
  return this.get(`/repos/${owner}/${repo}/git/ref/${ref}`, userOpts);
};

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param {object} payload
 * @param {string} payload.sha
 * @param {[boolean]} payload.force
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.updateRef = function (context, payload, userOpts) {
  // https://developer.github.com/v3/git/refs/#update-a-reference
  const {owner, repo, ref} = context;
  debug(`updateRef ${owner}/${repo} ${ref} -> ${payload.sha}`);
  return this.patch(`/repos/${owner}/${repo}/git/refs/${ref}`, payload, userOpts);
};

/**
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @param  {string} context.ref
 * @param  {object} [userOpts]
 * @return {promise}
 */
Github.prototype.deleteRef = function (context, userOpts) {
  debug(`deleteRef ${context.owner}/${context.repo} ${context.ref}`);
  // https://developer.github.com/v3/git/refs/#delete-a-reference
  const {owner, repo, ref} = context;
  return this.delete(`/repos/${owner}/${repo}/git/refs/${ref}`, userOpts);
};

module.exports = Github;
