'use strict';

const Agent = require('https').Agent;
const API = require('./API');
const defaults = require('lodash').defaults;
const inherits = require('util').inherits;

const agent = new Agent({
  keepAlive: true,
});

function Startrack(opts) {
  opts = opts || {};
  defaults(opts, {
    agent: agent,
    uri: 'https://st-api.yandex-team.ru/v2/'
  });

  API.call(this, opts);
}

inherits(Startrack, API);

/**
 * @param  {string}  issueId
 * @param  {object}  comment
 * @param  {string}  comment.text
 * @param  {array}   [comment.attachmentIds]
 * @param  {array}   [comment.summonees]
 * @return {promise}
 */
Startrack.prototype.createAComment = function (issueId, comment) {
  // https://wiki.yandex-team.ru/tracker/api/issues/comments/create/
  return this.post(`/issues/${issueId}/comments`, comment);
};

/**
 * @param  {string} issueId
 * @return {promise}
 */
Startrack.prototype.getIssue = function (issueId) {
  // https://wiki.yandex-team.ru/tracker/api/issues/get/
  return this.get(`/issues/${issueId}`);
};

/**
 * @param  {string} issueId
 * @return {promise}
 */
Startrack.prototype.getIssueComments = function (issueId) {
  // https://wiki.yandex-team.ru/tracker/api/issues/comments/
  return _fetchAllPages(this, `/issues/${issueId}/comments`);
};

function _fetchAllPages(startrack, endpoint) {
  const perPage = 250;

  const fetchPage = (page = 1, result = []) =>
    startrack.get(`${endpoint}?page=${page}&perPage=${perPage}`)
    .then(resp => {
      if (!Array.isArray(resp)) {
        return resp;
      }

      const currentResult = result.concat(resp);

      if (resp.length === perPage) {
        return _fetchAllPages(startrack, endpoint, page + 1, currentResult);
      }

      return currentResult;
    });

    return fetchPage();
}

/**
 * @return {promise}
 */
Startrack.prototype.getReleaseIssues = function () {
  const query = 'Queue: PI AND Status: new AND Type: Release "Sort By": Weight desc, Created desc';
  return this.get(`/issues?query=${encodeURIComponent(query)}`);
}

/**
 * @param  {string} issueId
 * @return {promise}
 */
Startrack.prototype.resolveIssue = function (issueId) {
  // https://wiki.yandex-team.ru/tracker/api/issues/transitions-execute/
  return this.post(`/issues/${issueId}/transitions/resolve/_execute`, {
    resolution: 'fixed',
  });
};

/**
 * @param  {string} issueId
 * @return {promise}
 */
Startrack.prototype.mergeIssue = function (issueId) {
  // https://wiki.yandex-team.ru/tracker/api/issues/transitions-execute/
  return this.post(`/issues/${issueId}/transitions/merge/_execute`, {
    resolution: 'fixed',
  });
};

/**
 * @private
 * @return {string}
 */
Startrack.prototype._buildAuthorizationHeader = function () {
  return 'OAuth ' + this._token;
};

module.exports = Startrack;
