'use strict';

const {
  chain,
  filter,
  find,
  includes,
  isEmpty,
  map,
  result,
  sum,
} = require('lodash');

const debug = require('debug')('krush:helper');

const CHARCODE = {
  COLON: '58',
  MINUS: '45',
  ONE: '49',
  PLUS: '43',
  THUMBS_DOWN: '55357:56398',
  THUMBS_UP: '55357:56397',
  WHITESPACE: '32',
};

const PATTERN = {
  NEGATIVE_CODE_ENDING: new RegExp(`${CHARCODE.COLON}:${CHARCODE.MINUS}:${CHARCODE.ONE}:${CHARCODE.COLON}$`),
  NEGATIVE_SMILE_ENDING: new RegExp(`${CHARCODE.THUMBS_DOWN}$`),
  POSITIVE_CODE_ENDING: new RegExp(`${CHARCODE.COLON}:${CHARCODE.PLUS}:${CHARCODE.ONE}:${CHARCODE.COLON}$`),
  POSITIVE_SMILE_ENDING: new RegExp(`${CHARCODE.THUMBS_UP}$`)
};

/**
 * @param  {object} commit
 * @param  {string} commit.sha
 * @param  {object} context
 * @param  {string} context.owner
 * @param  {string} context.repo
 * @return {object}
 */
function createCommitContext(commit, context) {
  return {
    owner: context.owner,
    repo: context.repo,
    sha: commit.sha,
  };
}

/**
 * @param  {array}  votes
 * @param  {array}  blacklist
 * @return {object}
 */
function estimateVotes(votes, blacklist) {
  if (isEmpty(blacklist)) {
    return sum(map(votes, 'estimation'));
  }

  return sum(map(filter(votes, vote => !includes(blacklist, vote.author)), 'estimation'));
}

/**
 * @param  {object} comment
 * @return {object}
 */
function extractCommentData(comment) {
  return {
    body: comment.body,
    date: new Date(comment.updated_at).getTime(),
    reviewer: result(comment, 'user.login'),
  };
};

/**
 * @param  {object} commit
 * @return {object}
 */
function extractCommitData(commit) {
  return {
    author: result(commit, 'author.login'),
    date: new Date(result(commit, 'commit.author.date')).getTime(),
    sha: commit.sha,
  };
}

/**
 * https://developer.github.com/guides/traversing-with-pagination/#basics-of-pagination
 *
 * @param  {string} partOfLink
 * @return {object}
 */
function extractPage(partOfLink) {
  return {
    rel: String(/rel="(.*?)"/.exec(partOfLink) || '').split(',').pop(),
    url: String(/<.*?>/.exec(partOfLink) || '').replace(/[<>]/g, ''),
  };
}

/**
 * @param  {string} branch
 * @return {string}
 */
function extractIssueId(branch) {
  const ticket = /^[A-Z]+\-\d+/.exec(branch);
  return ticket && String(ticket) || null;
}

/**
 * @param  {object} team
 * @return {object}
 */
function extractTeam(team) {
  return {
    name: team.slug,
    id: team.id,
  };
}

/**
 * @param  {array}  comments
 * @param  {string} login
 * @return {array}
 */
function filterCommentsByAuthor(comments, login) {
  return comments.filter(comment => result(comment, 'user.login') !== login);
}

/**
 * @param  {array} teams
 * @param  {array} allowedTeams
 * @return {array}
 */
function filterTeams(teams, allowedTeams) {
  return Array.isArray(allowedTeams) && allowedTeams.length > 0
    ? teams.map(extractTeam).filter(team => allowedTeams.indexOf(team.name) > -1)
    : teams.map(extractTeam);
}

/**
 * @param  {array}  commits
 * @return {object}
 */
function findLastCommit(commits) {
  return chain(commits)
    .map(extractCommitData)
    .filter('date')
    .last()
    .value();
}

/**
 * https://developer.github.com/guides/traversing-with-pagination/#basics-of-pagination
 *
 * @param  {string} link
 * @return {string}
 */
function findLastPage(link) {
  const lastPage = typeof link === 'string'
    && find(link.split(', ').map(extractPage), {rel: 'last'});

  return lastPage
    ? lastPage.url
    : null;
}

/**
 * @param  {object} teams
 * @param  {string} author
 * @return {string|null}
 */
function findTeam(teams, author) {
  for (var team in teams) {
    if (!includes(teams[team], author)) {
      continue;
    }

    return team;
  }

  return null;
}

/**
 * @param  {object} teams
 * @param  {object} repos
 * @param  {string} repo
 * @param  {string} owner
 * @param  {string} author
 * @return {array}
 */
function findTeammates(teams, repos, repo, owner, author) {
  const teamName = repos[`${owner}/${repo}`] || repos[repo];
  const team = teams[teamName];

  debug(`Finding teammates for ${teamName} and author ${author}`);
  debug(`Team: ${JSON.stringify(team)}`);

  if (!team) {
    return [];
  }

  if (Array.isArray(team[0])) {
    for (let group of team) {
        if (group.includes(author)) {
            return group;
        }
    }

    return [];
  }

  return team || [];
}

function encodeTextFragment(text) {
  return text.trim().substr(-4).split('').map(symbol => symbol.charCodeAt(0)).join(':');
}

/**
 * Хелпер для фильтрации сообщений вида ':+1:' и ':-1:'.
 *
 * @param  {object}  comment
 * @return {boolean}
 */
function isEstimationComment(comment) {
  const encodedBody = encodeTextFragment(comment.body);

  return PATTERN.NEGATIVE_CODE_ENDING.test(encodedBody) ||
    PATTERN.NEGATIVE_SMILE_ENDING.test(encodedBody) ||
    PATTERN.POSITIVE_CODE_ENDING.test(encodedBody) ||
    PATTERN.POSITIVE_SMILE_ENDING.test(encodedBody);
}

/**
 * @param  {array}  comments
 * @return {object}
 */
function listVotes(comments) {
  return chain(comments)
    .map(extractCommentData)
    .filter(isEstimationComment)
    .map(measureEstimation)
    .orderBy('date', 'desc')
    .uniqBy('reviewer')
    .value();
}

/**
 * @param  {object} extractedComment
 * @param  {string} extractedComment.body
 * @return {object}
 */
function measureEstimation(extractedComment) {
  const encodedBody = encodeTextFragment(extractedComment.body);
  const isPositiveEstimation = PATTERN.POSITIVE_CODE_ENDING.test(encodedBody) || PATTERN.POSITIVE_SMILE_ENDING.test(encodedBody);

  extractedComment.estimation = isPositiveEstimation ? 1 : -1;
  return extractedComment;
}

exports.createCommitContext = createCommitContext;
exports.estimateVotes = estimateVotes;
exports.extractCommentData = extractCommentData;
exports.extractCommitData = extractCommitData;
exports.extractIssueId = extractIssueId;
exports.extractPage = extractPage;
exports.extractTeam = extractTeam;
exports.filterCommentsByAuthor = filterCommentsByAuthor;
exports.filterTeams = filterTeams;
exports.findLastCommit = findLastCommit;
exports.findLastPage = findLastPage;
exports.findTeam = findTeam;
exports.findTeammates = findTeammates;
exports.isEstimationComment = isEstimationComment;
exports.listVotes = listVotes;
exports.measureEstimation = measureEstimation;
