'use strict';

const yaml = require('js-yaml');
const {readFile, writeFile} = require('fs');
const {resolve: resolvePath} = require('path');
const {get, set} = require('lodash');
const yargs = require('yargs');
const semver = require('semver');

function updateSpec(unit, box, imageTag) {
    const specPath = resolvePath(__dirname, './spec/krush.yaml');

    return new Promise(resolve => {
        console.log('Reading file');
        readFile(specPath, {encoding: 'utf-8'}, (err, data) => {
            if (err) {
                throw err;
            }

            const parsedSpec = yaml.load(data);
            const boxImageSpecPath = `spec.deploy_units.${unit}.images_for_boxes.${box}`;
            if (!get(parsedSpec, boxImageSpecPath)) {
                throw new Error(`Box or unit ${unit}.${box} is not defined`);
            }

            const tagPath = `${boxImageSpecPath}.tag`;
            const currentTag = get(parsedSpec, tagPath);
            if (semver.gt(currentTag, imageTag)) {
                throw new Error(`Current tag "${currentTag}" is newer than given "${imageTag}"`);
            }

            set(parsedSpec, tagPath, imageTag);
            writeFile(specPath, yaml.dump(parsedSpec), (err) => {
                if (err) {
                    throw err;
                }
                resolve();
            });
        });
    });
}

function main() {
    const {box, tag} = yargs()
        .option('box', {
            alias: 'b',
            default: 'krush.krush',
            description: '<deploy_unit>.<box>',
            required: true,
        })
        .option('tag', {
            alias: 't',
            required: true,
        })
        .parse(process.argv);

    console.log('main: %s@%s', box, tag);
    const [deployUnit, localBox] = box.split('.');
    return updateSpec(deployUnit, localBox, tag)
        .then(() => process.stdout.write(`Spec updated to ${box}@${tag}\n`));
}

main().catch(err => {
    process.stderr.write(err);
    process.exit(1);
});
