import yargs from 'yargs';

import {TelegramBot} from './interface/telegram';
import {ConsumerService} from './interface/types';
import {Service} from './interface/webhooks';

import * as sentry from './remote/sentry';
import * as startrek from './remote/startrek';
import * as slack from './remote/slack';
import * as telegram from './remote/telegram';

import {isInProduction} from './lib/utils';
import {loadSecrets, setSecretOrDie} from './lib/secrets';
import {config} from './config';
import {Sandbox} from "./remote/sandbox";
import {TvmManager} from "./lib/TvmManager";

// Interfaces
function startTelegramInterface(secret: string | null) {
    if (secret !== null) {
        const telegramBot = new TelegramBot(secret);
        telegramBot.start();
        registerServiceShutdown(telegramBot);
    } else if (isInProduction()) {
        throw new Error('Can\'t find Telegram token');
    } else {
        console.error('Can\'t find Telegram token. Telegram interface disabled.');
    }
}

function startServiceInterface(port: number) {
    const service = new Service(port);

    service.start();
    registerServiceShutdown(service);
}

function registerServiceShutdown(service: ConsumerService): void {
    process.on('SIGTERM', () => service.stop());
    process.on('SIGINT', () => service.stop());
}

const {port} = config;

function main() {
    const secrets = loadSecrets();

    const args = yargs
        .option('no-telegram', {
            type: 'boolean',
            description: 'Disable Telegram interface',
        })
        .option('no-webhooks', {
            type: 'boolean',
            description: 'Disable Webhooks interface',
        })
        .option('no-telegram-remote', {
            type: 'boolean',
            description: 'Disable Telegram remote',
        })
        .option('no-sentry-remote', {
            type: 'boolean',
            description: 'Disable Sentry remote',
        })
        .option('no-startrek-remote', {
            type: 'boolean',
            description: 'Disable Startrek remote',
        })
        .option('no-slack-remote', {
            type: 'boolean',
            description: 'Disable Slack remote',
        })
        .option('no-sandbox-remote', {
            type: 'boolean',
            description: 'Disable Sandbox remote',
        })
        .argv;

    // Remotes
    if (args.sentryRemote !== false) {
        setSecretOrDie(
            secrets.sentry,
            sentry.setToken,
            'Can\'t find Sentry token',
            'Can\'t find Sentry token. Sentry calls will fail.',
        );
    }

    if (args.startrekRemote !== false) {
        setSecretOrDie(
            secrets.startrek,
            startrek.setToken,
            'Can\'t find Startrek token',
            'Can\'t find Startrek token. Startrek calls will fail.',
        );
    }

    if (args.slackRemote !== false) {
        setSecretOrDie(
            secrets.slack,
            slack.setSlackUrl,
            'Can\'t find Slack URL',
            'Can\'t find Slack URL. Slack calls will fail',
        );
    }

    if (args.telegramRemote !== false) {
        setSecretOrDie(
            secrets.telegram,
            telegram.setToken,
            'Can\'t find Telegram URL',
            'Can\'t find Telegram URL. Telegram calls will fail',
        );
    }

    if (args.sandboxRemote !== false) {
        setSecretOrDie(
            secrets.sandbox,
            Sandbox.instance.setToken,
            'Can\'t find Sandbox token',
            'Can\'t find Sandbox token. Sandbox calls will fail',
        );
    }

    if (args.telegram !== false) {
        startTelegramInterface(secrets.telegram);
    }

    if (args.webhooks !== false) {
        startServiceInterface(port);
    }

    setSecretOrDie(
        secrets.tvm,
        TvmManager.setToken,
        'Can\'t find TVM Token',
        'Can\'t find TVM Token. TVM requests will fail',
    );
}

main();
