import Telegraf, {ContextMessageUpdate, Middleware} from 'telegraf';

import {isInProduction} from '../../lib/utils';
import {ServiceCommands} from './types';
import {serviceCommands as sentryCommands} from './service/sentry';
import {ConsumerService} from '../types';

const commands: ServiceCommands = {
    ...sentryCommands,
};

function withReplyOnCatch(call: Middleware<ContextMessageUpdate>) {
    return (ctx: ContextMessageUpdate) => {
        try {
            call(ctx);
        } catch(error) {
            ctx.replyWithMarkdown(`*Got error:*\n${error}`);
        }
    };
}

export class TelegramBot implements ConsumerService {
    _bot: Telegraf<ContextMessageUpdate>;

    constructor(token: string) {
        const bot = new Telegraf(token, {});

        bot.telegram.getMe().then((botInfo) => {
            bot.options.username = botInfo.username;
        });

        bot.start(withReplyOnCatch((ctx) => ctx.reply('Welcome!')));
        bot.help(withReplyOnCatch((ctx) => ctx.reply('Send me a sticker')));
        bot.on('sticker', withReplyOnCatch((ctx) => ctx.reply('👍')));
        bot.hears('hi', withReplyOnCatch((ctx) => ctx.reply('Hey there')));
        bot.hears('error', withReplyOnCatch((ctx) => {ctx.reply('Throwing error'); throw new Error('Ooops!!!')}));

        bot.hears('is prod', withReplyOnCatch((ctx) => { ctx.reply(`${isInProduction()}`)}));

        for (let [command, handler] of Object.entries(commands)) {
            bot.command(command, withReplyOnCatch(handler));
        }

        this._bot = bot;
    }

    start(): void {
        this._bot.launch();
    }

    stop(): void {
        this._bot.stop();
    }
}
