import {ContextMessageUpdate} from 'telegraf';

import {getIgnoreRules, setIgnoreRules} from '../../../../remote/sentry';
import {ServiceCommands} from '../../types';

export const serviceCommands: ServiceCommands = {
    getSentryIgnoreRules,
    addSentryIgnoreRules,
    removeSentryIgnoreRules,
};

/**
 * Removes telegram command from input. Result is not trimmed.
 * 
 * Example:
 * '/command argument' => ' argument'
 * 
 * @param line line to remove command from
 */
function removeCommand(line: string): string {
    return line.trimStart().replace(/^\/\w+/, '');
}

async function getSentryIgnoreRules(ctx: ContextMessageUpdate): Promise<void> {
    const {reply} = ctx;
    let currentRules;

    try {
        currentRules = await getIgnoreRules();
    } catch (error) {
        reply(`Error: Can't get ignore rules: ${error}`);
        return;
    }

    reply(`Ignore rules:\n${currentRules}`);
}

async function addSentryIgnoreRules(ctx: ContextMessageUpdate): Promise<void> {
    const {reply} = ctx;
    const text = ctx.message?.text;
    let currentRules, nextRules;

    if (text === undefined) {
        return;
    }

    const newRule = removeCommand(text).trim();

    try {
        currentRules = await getIgnoreRules();
    } catch (error) {
        reply(`Error: Can't get ignore rules: ${error}`);
        return;
    }

    const newRules = [currentRules, newRule].join('\n');

    try {
        nextRules = await setIgnoreRules(newRules);
    } catch (error) {
        reply(`Error: Can't set ignore rules: ${error}`);
        return;
    }

    reply(`New ignore rules:\n${nextRules}`);
}

async function removeSentryIgnoreRules(ctx: ContextMessageUpdate): Promise<void> {
    const {reply} = ctx;
    const text = ctx.message?.text;
    let currentRules, nextRules;

    if (text === undefined) {
        reply(`Error: Can't get text from message.`);
        return;
    }

    const removeRule = removeCommand(text).trim();

    try {
        currentRules = await getIgnoreRules();
    } catch (error) {
        reply(`Error: Can't get ignore rules: ${error}`);
        return;
    }

    const newRules = currentRules
        .split('\n')
        .map(item => item.trim())
        .filter(item => item !== removeRule)
        .join('\n');

    try {
        nextRules = await setIgnoreRules(newRules);
    } catch (error) {
        reply(`Error: Can't set ignore rules: ${error}`);
        return;
    }

    reply(`New ignore rules:\n${nextRules}`);
}
