import express, {Express} from 'express';
import {Server as HttpServer} from 'http';
import {ConsumerService} from '../types';
import path from 'path';

import {HttpMethod, ServiceHandler} from './types';
import {serviceHandlers as e2eHandlers} from './service/e2e';
import {serviceHandlers as helloHandlers} from './service/hello';
import {serviceHandlers as githubHandlers} from './service/github';
import {serviceHandlers as notificationsHandlers} from './service/notifications';
import {serviceHandlers as wbsHandlers} from './service/wbs';
import {serviceHandlers as timelineHandlers} from './service/timeline';
import {serviceHandlers as loggerHandlers} from './service/logger';
import {HttpError} from './errors';
import {config} from './../../config';

const handlers: ServiceHandler[] = [
    ...helloHandlers,
    ...notificationsHandlers,
    ...wbsHandlers,
    ...timelineHandlers,
    ...githubHandlers,
    ...e2eHandlers,
    ...loggerHandlers,
];

const allowedDomains = [
    '.yandex-team.ru',
    '.yandex.ru',
]

export class Service implements ConsumerService{
    private app: Express;
    private port: number;
    private server: HttpServer;

    constructor(port: number) {
        const app = express();
        app.use((req, res, next) => {
            const origin = req.headers?.origin || req.hostname;
            const corsRegExp = new RegExp(`${allowedDomains.join('|')}$`);
            if (corsRegExp.test(origin.toString())) {
                res.header('Access-Control-Allow-Origin', origin);
                res.header('Access-Control-Allow-Methods', 'GET, HEAD, OPTIONS, POST, PUT');
                res.header('Access-Control-Allow-Headers', 'Origin, Accept, X-Requested-With, Content-Type');
                res.header('Cache-Control', 'no-cache');
            };
            next();
        });
        
        const {jsonLimit} = config;
        app.use(express.json({limit: jsonLimit})); // for parsing application/json
        app.use(express.urlencoded({extended: true})); // for parsing application/x-www-form-urlencoded
        app.use('/assets/', (req, res, next) => {
                req.url = req.url.replace(/^\/assets\//, '');
                next();
            },
            express.static(path.resolve(__dirname, '../../../build'), {
                fallthrough: false,
            })
        );

        for (let {method, path, handler} of handlers) {
            if (method === HttpMethod.GET) {
                app.get(path, handler);
            } else if (method === HttpMethod.POST) {
                app.post(path, handler);
            } else if (method === HttpMethod.PUT) {
                app.put(path, handler);
            } else if (method === HttpMethod.DELETE) {
                app.delete(path, handler);
            }
        }

        app.use((error: Error, req: express.Request, res: express.Response, next: express.NextFunction) => {
            console.error(error);
            res.status(error instanceof HttpError ? error.code : 500);
            res.send(error.message);
        });

        this.app = app;
        this.port = port;
    }

    public start(): void {
        this.server = this.app.listen(this.port, () => {
            console.log(`Krusher runs on port ${this.port}`)
        });
    }

    public stop(): void {
        this.server.close();
    }
}
