import {Request, RequestHandler, Response, NextFunction} from "express";
import {Sandbox} from "../../../../remote/sandbox";
import {SandboxTaskType} from "../../../../remote/sandbox/types";
import {HttpBadRequestError} from "../../errors";

interface InputParameters extends Record<string, string | string[] | undefined>{
    frontend_branch: string;
    java_branch_or_tag?: string;
    perl_branch_or_tag?: string;
    hermione_report_resource_id: string;
    to_update_paths: string[];
}

export class UpdateScreensHandler {
    public runMakeUpdateRefsPrTask: RequestHandler = async (req: Request, res: Response, next: NextFunction): Promise<void> => {
        try {
            this.checkOrigin(req, res);
            const inputParams = this.parseRequestBody(req.body);

            const {body} = await Sandbox.instance.createTask<InputParameters>(
                SandboxTaskType.PARTNER_E2E_MAKE_UPDATE_REFS_PR,
                inputParams,
                `Update screens for branch ${inputParams.frontend_branch}`
            );
            const {id} = body;
            await Sandbox.instance.startTasksByIds([id]);

            const taskUrl = Sandbox.instance.getSandboxTaskUrl(id);

            res.status(201)
            res.set('Access-Control-Expose-Headers', 'Location');
            res.set('Location', taskUrl);
            res.send('Created');
        } catch (error) {
            next(error);
        }
    };

    private parseRequestBody(requestBody: any): InputParameters {
        if (typeof requestBody !== 'object' || requestBody === null) {
            throw new HttpBadRequestError('Request body is not an object');
        }

        const {
            frontend_branch,
            java_branch_or_tag,
            perl_branch_or_tag,
            resource_id: resourceId,
            items,
        } = requestBody;

        if (typeof frontend_branch !== 'string') {
            throw new HttpBadRequestError('Branch parameter is not a string');
        }

        if (!resourceId) {
            throw new HttpBadRequestError('There is no resource id');
        }

        if (!Array.isArray(items)) {
            throw new HttpBadRequestError('Item parameter is not array');
        }

        return {
            frontend_branch,
            java_branch_or_tag,
            perl_branch_or_tag,
            hermione_report_resource_id: resourceId,
            to_update_paths: items
        }
    }

    // TODO по-хорошему тут нужно middleware https://www.npmjs.com/package/cors
    private checkOrigin(req: Request, res: Response): void {
        const origin = req.get('origin') || UpdateScreensHandler.DEFAULT_ORIGIN;
        // здесь можно добавить whitelist: [DEFAULT_ORIGIN, http://localhost:*]

        res.set('Access-Control-Allow-Origin', origin);
        res.set('Vary', 'Origin');
    }

    static readonly DEFAULT_ORIGIN = 'https://sandbox.yandex-team.ru';
}
