import {RequestBody, Target} from "./types";
import {isValidTicketId} from "../../../../lib/helpers";
import {addCommentToReleaseChannel as notifySlack} from "../../../../remote/slack";
import {createComment as notifyStartrek} from "../../../../remote/startrek";
import moment from "moment";
import {WORKING_TIME} from "./constants";
import {addCommentToReleaseChat as notifyTelegram, Chats} from "../../../../remote/telegram";
import {Request, Response} from "express";

export class NotificationHandler {

	constructor() {
		this.parseRequestBody = this.parseRequestBody.bind(this);
		this.notify = this.notify.bind(this);
		this.release = this.release.bind(this);
		this.direct = this.direct.bind(this);
	}
	private parseRequestBody(requestBody: any): RequestBody {
		if (typeof requestBody !== 'object' || requestBody === null) {
			throw new Error('Request body is not an object');
		}

		const {
			disableNotifications,
			message,
			targets,
			ticketId: rawTicketId,
		} = requestBody;
		const ticketId = rawTicketId.trim();

		if (typeof message !== 'string') {
			throw new Error('Message to send is not a string');
		}

		if (typeof ticketId !== 'string' || !isValidTicketId(ticketId)) {
			throw new Error('Ticket Key is not valid');
		}

		return {
			disableNotifications,
			message,
			targets,
			ticketId,
		}
	}

	private notify(
		target: Target,
		message: string,
		ticketId?: string,
		disableNotifications?: boolean,
	): Promise<any> {
		switch (target) {
			case 'slack':
				return notifySlack(message)
					.catch(error => console.error(`Failed to send Slack message: ${error}`));

			case 'startrek':
				if (ticketId) {
					return notifyStartrek(ticketId, message)
						.catch(error => console.error(`Failed to send Startrek message: ${error}`));
				}
				break;

			case 'telegram':
				return this.notifyTelegram(message, disableNotifications, Chats.release_pi)
					.catch(error => console.error(`Failed to send Telegram message: ${error}`))
		}

		return new Promise((resolve) => resolve());
	}

	private async notifyTelegram(message: string, disableNotifications?: boolean, target?: string | Chats): Promise<any> {
		if (disableNotifications &&
			moment().isBetween(moment(WORKING_TIME[0], 'hh:mm'), moment(WORKING_TIME[1], 'hh:mm'))
		) {
			disableNotifications = false;
		}
		if (!target) {
			throw Error('No target specify');
		}

		return notifyTelegram(message, target as unknown as Chats, disableNotifications);
	}

	async release(req: Request, res: Response): Promise<any> {
		let parsedBody: RequestBody | null = null;

		try {
			parsedBody = this.parseRequestBody(req.body);
		} catch (errorMessage) {
			console.error(errorMessage);
			res.sendStatus(400);
			res.send(errorMessage);
			return;
		}

		if (parsedBody) {
			const {
				disableNotifications,
				message,
				ticketId,
			} = parsedBody;
			const requests: Promise<any>[] = [];
			let {targets} = parsedBody;

			if (!Array.isArray(targets) || targets.length === 0) {
				targets = [Target.slack, Target.startrek, Target.telegram];
			}

			targets.forEach((target: Target) => {
				requests.push(this.notify(target, message, ticketId, disableNotifications));
			});

			const requestStatus = await Promise.all(requests);

			res.send(requestStatus);

			return requestStatus;
		}
	}

	async direct(req: Request, res: Response): Promise<any> {
		const {
			disableNotifications,
			message,
			targets,
		} = req.body;
		if (!Array.isArray(targets)) {
			res.status(400);
			res.send('targets must be array');
			return;
		}

		const requests: Promise<any>[] = [];
		targets.forEach((target: string) => {
			requests.push(this.notifyTelegram(message, disableNotifications, target));
		});

		let requestStatus;
		try {
			requestStatus = await Promise.all(requests);
		} catch (ex) {
			requestStatus = {
				exception: ex.toString(),
			};
		}
		res.send(requestStatus);

		return requestStatus;
	}
}
