import {Request, Response} from 'express';

import {loadIndex as openForm} from '../lib/loadIndex'
import {getUniqueStringArray, isValidTicketId, parseTicketId} from '../../../../lib/helpers';
import {getTimelineTasks, filterTimelineTasks, enhanceTimelineTasksWithDates} from '../../../../remote/startrek/timeline';

import {ServiceHandler, HttpMethod} from '../../types';
import {TimelineRequestBody as RequestBody, TimelineError, TaskGroup, TaskGroupWithNest, TaskGroups, TaskItem, TaskItemType} from './types';
import {TicketNodeWithChildrenIds as TicketNode, TimelineTicket} from '../../../../remote/startrek/timeline/types';

function parseRequestBody(requestBody: any): RequestBody | TimelineError {
    if (typeof requestBody !== 'object' || requestBody === null) {
        return {
            isError: true,
            error: {
                general: 'Тело запроса не является объектом',
            }
        };
    }

    const {ticketId: rawTicketId, filterTags: rawFilterTags} = requestBody;
    const ticketId = rawTicketId.trim();

    if (!isValidTicketId(ticketId)) {
        return {
            isError: true,
            error: {
                ticketId: 'Ключ тикета не валидный',
            }
        };
    }

    // по умолчанию фильтруем по беклогу
    const defaultFilterTags = ['backlog'];
    let filterTags: string[] = [];
    if (rawFilterTags) {
        filterTags = getUniqueStringArray(rawFilterTags).concat(...defaultFilterTags);
    }

    return {
        isError: false,
        ticketId,
        filterTags,
    }
}

function getTaskGroup(node: TicketNode): TaskGroup {
    const {id: ticketId} = parseTicketId(node.id);

    return {
        id: ticketId,
        treeLevel: node.nest + 1,
        content: node.id,
    }
}

function getTaskGroupWithNest(currTask: TicketNode, tasks: TimelineTicket[]): TaskGroupWithNest {
    const childTicketIds = currTask.children.filter((childId) => {
        const taskWithId = tasks.find((task) => task.id === childId);

        return Boolean(taskWithId);
    });

    const numericChildTicketIds = childTicketIds.map((childId) => {
        return parseTicketId(childId).id;
    });

    const {id: ticketId} = parseTicketId(currTask.id);


    return {
        id: ticketId,
        treeLevel: currTask.nest + 1,
        content: currTask.id,
        // может произойти коллизия при одинаковых номерах задач из разных очередей
        nestedGroups:numericChildTicketIds,
    };
}

function reduceTreeByGroups(tasks: TimelineTicket[]): TaskGroups {
    const initial: TaskGroups = {
        groupsWithNest: [],
        groups: [],
    };

    const result = tasks.reduce((acc: TaskGroups, task: TicketNode) => {
        if (!task.isError && task.children.length > 0) {
            const nextGroupsWithNest = acc.groupsWithNest.concat(getTaskGroupWithNest(task, tasks));

            return {
                groups: acc.groups,
                groupsWithNest: nextGroupsWithNest,
            }
        } else {
            const nextGroups = acc.groups.concat(getTaskGroup(task));

            return {
                groups: nextGroups,
                groupsWithNest: acc.groupsWithNest,
            };
        }
    }, initial);

    return result;
}

function processTreeByItems(nodes: TimelineTicket[]): TaskItem[] {
    const result = nodes.map((node) => {
        const {id: ticketId} = parseTicketId(node.id);

        let endTime = '';
        if (node.isError) {
            endTime = node.end;
        } else {
            endTime = node.isFinished ? node.currentStatusStartTime : node.end;
        }

        const description = node.isError ? `${node.id} - [no_details]`: `${node.id}: ${node.summary}`;

        const currentNodeItem: TaskItem = {
            id: ticketId,
            group: ticketId,
            content: description,
            start: node.start,
            end: endTime,
            title: description,
            type: TaskItemType.range,
            isClosed: node.isError ? false : node.isFinished,
        };

        return currentNodeItem;
    });

    return result;
}

async function makeTimeline(req: Request, res: Response) {
    const parsedBody = parseRequestBody(req.body);

    if (parsedBody.isError) {
        console.error(String(parsedBody.error));
        res.send(parsedBody.error);
    }

    if (!parsedBody.isError) {
        const {ticketId, filterTags} = parsedBody;

        const timelineTasks = await getTimelineTasks(ticketId)
            .then((tree) => {
                const filteredTree = filterTimelineTasks(tree, filterTags);
                return enhanceTimelineTasksWithDates(filteredTree);
            })
            .then((result) => result)
            .catch((error: string) => {
                console.error(`Failed to get work breakdown structure: ${error}`);
                res.send({
                    error:  String(error),
                });
            });

        if (timelineTasks) {
            const tasksItems = processTreeByItems(timelineTasks);
            const tasksGroups = reduceTreeByGroups(timelineTasks);


            res.send({
                tasksGroups,
                tasksItems,
            });
        }
    }
}

export const serviceHandlers: ServiceHandler[] = [
    {
        path: '/timeline/make',
        method: HttpMethod.POST,
        handler: makeTimeline,
    },
    {
        path: '/timeline/form',
        method: HttpMethod.GET,
        handler: openForm,
    },
];
