import {Request, Response} from 'express';

import {getFilteredTicketsDescription, getWbs} from '../../../../remote/startrek/wbs';
import {getTicketDescription, updateDescription} from '../../../../remote/startrek';
import {loadIndex as openForm} from '../lib/loadIndex'
import {getUniqueStringArray, isValidTicketId} from '../../../../lib/helpers';
import {calculateBudget} from '../../../../lib/budget/Budget';
import {nodeToMarkdown} from '../../../../lib/formatters/nodeToMarkdown';
import {updateTextWithAddition} from '../../../../lib/commonFunctions'

import {ServiceHandler, HttpMethod} from '../../types';
import {WbsRequestBody as RequestBody, WbsError, FilteredTickets, BudgetPreset} from './types';
import {TicketNodeWithChildren as TicketNode} from "../../../../remote/startrek/wbs/types";

const START_TOKEN = '=== Иерархическая структура работ\n';
const END_TOKEN = '=====\n';

function parseRequestBody(requestBody: any): RequestBody | WbsError {
    if (typeof requestBody !== 'object' || requestBody === null) {
        const previewRequestBody = String(requestBody).length > 100
            ? String(requestBody).slice(0, 100)
            : String(requestBody);

        return {
            isError: true,
            error: {
                general: 'Тело запроса не является объектом',
                details: previewRequestBody,
            }
        };
    }

    const {
        ticketId: rawTicketId,
        outputTicketId: rawOutputTicketId = '',
        filterTags: rawFilterTags,
        budgetPreset: rawBudgetPreset,
    } = requestBody;

    const ticketId = rawTicketId.trim();
    const outputTicketId = rawOutputTicketId.trim();

    if (!isValidTicketId(ticketId)) {
        return {
            isError: true,
            error: {
                ticketId: `Ключ тикета ${ticketId} не валидный`,
            }
        };
    }


    if (outputTicketId && !isValidTicketId(outputTicketId)) {
        return {
            isError: true,
            error: {
                outputTicketId: `Ключ тикета ${outputTicketId} не валидный`,
            }
        };
    }

    let filterTags: string[] = [];
    if (rawFilterTags) {
        filterTags = getUniqueStringArray(rawFilterTags);
    }

    let budgetPreset = undefined;
    if (rawBudgetPreset && Object.values(BudgetPreset).includes(rawBudgetPreset.toLowerCase())) {
        budgetPreset = rawBudgetPreset.toLowerCase();
    }

    return {
        isError: false,
        ticketId,
        outputTicketId,
        filterTags,
        budgetPreset,
    }
}

function getMarkdownStructure(tree: TicketNode, filteredTickets: FilteredTickets, filterTags: string[], budgetPreset?: BudgetPreset) {
    const markdownTree = nodeToMarkdown(tree);
    const markdownFilteredTickets = getFilteredTicketsDescription(filteredTickets, tree.id);

    let markdownBudget = '';
    if (budgetPreset) {
        markdownBudget = calculateBudget(tree, budgetPreset, filterTags);
    }

    return `${markdownTree}${markdownFilteredTickets}${markdownBudget}`;
}

async function makeWbs(req: Request, res: Response): Promise<any> {
    const parsedBody = parseRequestBody(req.body);

    if (parsedBody.isError) {
        console.error(JSON.stringify(parsedBody.error));
        res.send(parsedBody);
    }

    if (!parsedBody.isError) {
        const {ticketId, outputTicketId, filterTags, budgetPreset} = parsedBody;

        const description = await getWbs(ticketId, filterTags)
            .then(([treeStructure, filteredTickets]) => getMarkdownStructure(treeStructure, filteredTickets, filterTags, budgetPreset))
            .catch((error: string) => {
                console.error(`Failed to get work breakdown structure: ${error}`);
                res.send({
                    error: {
                        ticketId: String(error),
                    },
                });
            });

        if (description) {
            res.send(description);

            if (outputTicketId) {
                const test = await getTicketDescription(outputTicketId)
                    .then((outputTicketDescription: string) => {
                        const updatedDescription = updateTextWithAddition(outputTicketDescription, description, START_TOKEN, END_TOKEN);
                        return updateDescription(outputTicketId, updatedDescription);
                    })
                    .then((requestResult) => {
                        res.sendStatus(requestResult.status);
                    })
                    .catch((error) => {
                        console.error(`Failed to get output ticket description: ${error}`);
                        res.send({
                            error: {
                                outputTicketId: String(error),
                            },
                        });
                    })
            }
        }
    }
}

export const serviceHandlers: ServiceHandler[] = [
    {
        path: '/wbs/form',
        method: HttpMethod.GET,
        handler: openForm,
    },
    {
        path: '/wbs/make',
        method: HttpMethod.POST,
        handler: makeWbs,
    },
];
