import {TicketNodeWithChildren as TreeNode} from '../../remote/startrek/wbs/types';
import {BudgetTeamDefinition, BudgetTeamItem, BudgetClass} from './types';
import {budgetSettings} from './budgetSettings';
import {logError, stringContainsOneOf, stringDoesntContainAnyOf, arrayContainsOneOf} from '../commonFunctions';
import {budgetToMarkdown} from '../formatters/budgetToMarkdown';
import {BudgetPreset} from '../../interface/webhooks/service/wbs/types';
import {parseTicketId} from "../helpers";

export default class Budget implements BudgetClass {
    teamFilter: string[];
    excludeTags: string[];
    teams: Record<string, BudgetTeamItem>;
    teamDefinitions: BudgetTeamDefinition[];
    originalSP: number;

    constructor(
        teamFilter: string[],
        teamDefinitions: BudgetTeamDefinition[],
        excludeTags: string[],
        originalSP: number,
    ) {
        this.teamFilter = teamFilter;
        this.excludeTags = excludeTags.length > 0 ? excludeTags : [];
        this.teamDefinitions = teamDefinitions;
        this.teams = {};
        this.originalSP = originalSP;
    }

    private addTeam(team: string): BudgetTeamItem {
        const newTeamItem: BudgetTeamItem = {
            name: team,
            spentSP: 0,
            estimatedSP: 0,
            numEndTasksWithoutEstimation: 0,
            tags: [],
        };

        this.teams = {
            ...this.teams,
            [team]: newTeamItem,
        };

        return newTeamItem;
    }

    private updateTeam(nextTeam: BudgetTeamItem): void {
        this.teams = {
            ...this.teams,
            [nextTeam.name]: nextTeam,
        }
    }

    defineTeam(node: TreeNode): string {
        if (!(typeof node === 'object' && node.id))
            logError('Can not define team, node is not of correct object type');

        const {queue} = parseTicketId(node.id);
        if (!queue) {
            return node.id
        }

        const nodeTitle = node.summary || '';
        const nodeTags = node.tags || [];

        const team = this.teamDefinitions.find((definedTeam) => (
                definedTeam.queues.includes(queue) &&
                arrayContainsOneOf(nodeTags, definedTeam.tags) &&
                stringContainsOneOf(nodeTitle, definedTeam.titleContains) &&
                stringDoesntContainAnyOf(nodeTitle, definedTeam.titleDoesntContain)
        ));

        if (team) {
            return team.name;
        }

        return queue;
    }

    calcBudgetRecursively(node: TreeNode): void {
        const teamName = this.defineTeam(node);

        let team = this.teams[teamName];

        if (!team) {
            team = this.addTeam(teamName);
        }

        const printTeamBudget = this.teamFilter.length === 0 || this.teamFilter.includes(team.name);

        const nodeTags = node.tags || [];

        if (printTeamBudget && !arrayContainsOneOf(nodeTags, this.excludeTags)) {
            if (node.storyPoints) {
                if (node.isFinished) {
                    team.spentSP += node.storyPoints;
                } else {
                    team.estimatedSP += node.storyPoints;
                }
            } else {
                if (!node.isFinished && node.children.length === 0) {
                    team.numEndTasksWithoutEstimation += 1;
                }
            }
            this.updateTeam(team);
        }

        if (node.children) {
            for (let i = 0; i < node.children.length; i++) {
                const child = node.children[i];

                if (!child.isError) {
                    this.calcBudgetRecursively(node.children[i] as TreeNode);
                }
            }
        }
    }

    // Функция считает бюджет по группам разработки
    calcBudget = (tree: TreeNode): void => this.calcBudgetRecursively(tree);
}

// Функция расчитывает бюджет и возвращает его отформатированным
export const budgetFromTree = (tree: TreeNode, preset: BudgetPreset, excludeTags?: string[]): BudgetClass => {
    const finalExcludeTags = excludeTags && excludeTags.length
        ? excludeTags
        : budgetSettings[preset].excludeFromBudgetTags;

    const originalSP: number = tree && tree.originalStoryPoints
        ? tree.originalStoryPoints
        : 0;

    const budget = new Budget(
        budgetSettings[preset].teamFilterForBudget,
        budgetSettings[preset].teamDefinitions,
        finalExcludeTags,
        originalSP,
    );

    budget.calcBudget(tree);
    return budget;
};

export const calculateBudget = (tree: TreeNode, preset: BudgetPreset, excludeTags: string[]): string => {
    const budget = budgetFromTree(tree, preset, excludeTags);

    return budgetToMarkdown(budget);
};
