import {TicketNodeWithChildren as TreeNode, WbsTicket} from '../../remote/startrek/wbs/types';

/**
 * @param {TreeNode} param0
 * @returns boolean
 */
const isFinished = ({isFinished}: TreeNode): boolean | undefined => isFinished;

/**
 * @param {TreeNode} param0
 * @returns boolean
 */
const hasChildren = ({children}: TreeNode): boolean => (children && children.length > 0);

/**
 * Фабрика обёрток вида XX+text+XX
 * @param {string} token
 * @returns {function(string):string}
 */
const decorateWith = (text: string, token: string): string => `${token}${text}${token}`;


/** добавляет пробел между маркером списка и текстом */
const addInitialSpace = (text: string): string => ' ' + text;

const getQueue = (key: string): string => key.substr(0, key.indexOf('-'));
const getNumInQueue = (key: string): string => key.substr(key.indexOf('-') + 1, key.length);
const compareNodesForSort = (a: TreeNode, b: TreeNode): number => {
    let result = 1;
    result = a.isFinished && !b.isFinished ? -1 : 1;
    if (a.isFinished === b.isFinished) {
        result = a.id < b.id ? -1 : 1;
        if (a.id && b.id && getQueue(`${a.id}`) === getQueue(`${b.id}`))
            result = +getNumInQueue(`${a.id}`) < +getNumInQueue(`${b.id}`) ? -1 : 1;
    }
    return result;
};

/** закрытие тикеты с детьми сворачивает */
const collapseNode = (text: string): string => `+ ${text}`;

/** @param {TreeNode} node */
const markFinished = (text: string, node: TreeNode): string => (isFinished(node) ? decorateWith(text, '--') : text);

/** @param {TreeNode} node  */
const markHasChildren = (text: string, node: TreeNode): string => (hasChildren(node) ? decorateWith(text, '**') : text);

/** @param {TreeNode} node  */
const markFinishedSubtree = (text: string, node: TreeNode): string => (isFinished(node) && hasChildren(node) ? collapseNode(text) : text);

/** @param {TreeNode} node  */
const addSprint = ({sprint}: TreeNode): string => (sprint ? ` [sprint:${sprint.display}]` : '');

/** @param {TreeNode} node  */
const addSP = (node: TreeNode): string => (node.storyPoints ? ` [${node.storyPoints}SP]` : '');

/** @param {TreeNode} node  */
const addStatus = (node: TreeNode): string =>
    node.status && node.status.display ? ` [${node.status.display}]` : '';

/** @param {TreeNode} node  */
const addTitle = (node: TreeNode): string => (node.summary ? `: ${node.summary}` : '');

const getKey = (node: WbsTicket): string => {
    if (node.isError) {
        return node.id;
    }

    return (isFinished(node) ? `((${node.id}))` : node.id);
};

/**
 * Вывод узла в виде markdown
 * @param {TreeNode} node
 * @returns string
 */
export const getLineForNode = (node: WbsTicket): string => {
    let result = '';

    if (node.isError) {
        result += getKey(node);
        result += ' [no_details]';

        return addInitialSpace(result);
    }

    result += getKey(node);
    result += addTitle(node);
    result += addStatus(node);
    result += addSP(node);
    result += addSprint(node);
    result = markHasChildren(result, node);
    result = markFinished(result, node);
    result = addInitialSpace(result);
    result = markFinishedSubtree(result, node);
    return result;
};

/**
 * Вывод узла и его детей в виде markdown
 * @param {TreeNode} node
 * @param {string} incomingPrefix
 * @returns string
 */
const formatRecursively = (node: WbsTicket, incomingPrefix: string): string => {
    let result = '';

    if (node.isError) {
        return result;
    }

    if (node.children.length) {
        const prefix = '  ' + incomingPrefix;
        const children = node.children.sort((a: TreeNode, b: TreeNode): number => compareNodesForSort(a, b));

        for (let i = 0; i < children.length; i++) {
            const child = children[i];
            result += `${prefix}${getLineForNode(child)}\n`;

            if (child.isError) {
                continue;
            }

            if (child.children.length) {
                result += formatRecursively(child, prefix);
            }
        }
    }

    return result;
};

/**
 * Компиляция полного текста в виде markdown
 * @param {TreeNode} node
 * @returns string
 */
export const nodeToMarkdown = (node: TreeNode): string => {
    return `=== ((${node.id})) ${node.summary}\n${formatRecursively(node, '1.')}`;
};
