import superagent, {Response} from 'superagent';

function parseErrorResponse(body: any): string {
    let reply;

    try {
        reply = JSON.parse(JSON.stringify(body));
    } catch (error) {
        return `Can\'t parse reply: ${error}`;
    }

    const {errorMessages = null} = reply;

    if (typeof errorMessages === 'object') {
        return JSON.stringify(errorMessages);
    }

    if (!Array.isArray(errorMessages)) {
        return 'Unexpected reply, errorMessages is not an array';
    }

    for (let errorMessage of errorMessages) {
        if (typeof errorMessage !== 'string') {
            return `Unexpected error type in errors list: ${typeof errorMessage}`;
        }
    }

    return errorMessages.join(', ');
}

function catchRequestErrors(agentError: { response: { body: any; }; }) {
    if (agentError?.response) {
        const errorMessage = parseErrorResponse(agentError.response.body);
        throw new Error(errorMessage);
    }
    throw agentError;
}

export async function performOAuthGetRequest<Result>(
    url: string,
    thenCallback: (body: any) => Result,
    oauthToken: string
): Promise<Result | void>{

    return superagent
        .get(url)
        .set('Authorization', `OAuth ${oauthToken}`)
        .then((res) => thenCallback(res?.body))
        .catch(catchRequestErrors);
}

export async function performOAuthPostRequest(
    url: string,
    requestBody: Record<string, unknown>,
    oauthToken: string
): Promise<Response | void>{

    return superagent
        .post(url)
        .set('Authorization', `OAuth ${oauthToken}`)
        .send(requestBody)
        .catch(catchRequestErrors);
}

export async function performOAuthPutRequest<Result>(
    url: string,
    requestBody: any,
    oauthToken: string,
    ): Promise<Response | void> {
    return superagent
        .put(url)
        .set('Authorization', `OAuth ${oauthToken}`)
        .send(requestBody)
        .catch(catchRequestErrors);
}

export async function performOAuthPatchRequest(
    url: string,
    requestBody: any,
    oauthToken: string
): Promise<Response | void> {
    return superagent
        .patch(url)
        .set('Authorization', `OAuth ${oauthToken}`)
        .send(requestBody)
        .catch(catchRequestErrors);
}
