import fs from 'fs';
import {isInProduction} from './utils';

const TELEGRAM_TOKEN_FILENAME = '.telegram.token';
const TELEGRAM_TOKEN_ENVVAR_NAME = 'TELEGRAM_TOKEN';
const SENTRY_TOKEN_FILENAME = '.sentry.token';
const SENTRY_TOKEN_ENVVAR_NAME = 'SENTRY_TOKEN';
const STARTREK_TOKEN_FILENAME = '.startrek.token';
const STARTREK_TOKEN_ENVVAR_NAME = 'STARTREK_TOKEN';
const SLACK_TOKEN_FILENAME = '.slack.token';
const SLACK_TOKEN_ENVVAR_NAME = 'SLACK_TOKEN';
const SANDBOX_TOKEN_FILENAME = '.sandbox.token';
const SANDBOX_TOKEN_ENVVAR_NAME = 'SANDBOX_TOKEN';
const TVM_TOKEN_FILE = '.tvm.token';
const TVM_TOKEN_ENV = 'TVMTOOL_LOCAL_AUTHTOKEN';

export interface Secrets {
    telegram: string | null;
    sentry: string | null;
    startrek: string | null;
    slack: string | null;
    sandbox: string | null;
    tvm: string | null;
}

function getSecretFromFileOrEnv(filename: string, envvar: string): string | null {
    let result = null;

    if (fs.existsSync(filename)) {
        const token = fs.readFileSync(filename).toString().trim();

        if (token.length > 0) {
            result = token;
        }
    }

    const token = process.env[envvar];

    if (token && token.length > 0) {
        result = token;
    }

    return result;
}

export function loadSecrets(): Secrets {
    const telegram = getSecretFromFileOrEnv(TELEGRAM_TOKEN_FILENAME, TELEGRAM_TOKEN_ENVVAR_NAME);
    const sentry = getSecretFromFileOrEnv(SENTRY_TOKEN_FILENAME, SENTRY_TOKEN_ENVVAR_NAME);
    const startrek = getSecretFromFileOrEnv(STARTREK_TOKEN_FILENAME, STARTREK_TOKEN_ENVVAR_NAME);
    const slack = getSecretFromFileOrEnv(SLACK_TOKEN_FILENAME, SLACK_TOKEN_ENVVAR_NAME);
    const sandbox = getSecretFromFileOrEnv(SANDBOX_TOKEN_FILENAME, SANDBOX_TOKEN_ENVVAR_NAME);
    const tvm = getSecretFromFileOrEnv(TVM_TOKEN_FILE, TVM_TOKEN_ENV);

    return {
        telegram,
        sentry,
        startrek,
        slack,
        sandbox,
        tvm,
    };
}

export function setSecretOrDie(secret: string | null, secretSetter: (secret: string) => void, throwErrorMessage: string, logErrorMessage: string) {
    if (secret !== null) {
        secretSetter(secret);
    } else if (isInProduction()) {
        throw new Error(throwErrorMessage);
    } else {
        console.error(logErrorMessage);
    }
}
