import superagent from 'superagent';
import {Schema, SchemaType} from './schema';
import config from './config.json';
import {IData, IRequestData, IConfig} from './types';
import {HttpMethod} from '../../interface/webhooks/types';

export class ClickhouseClient {
    private config: IConfig;
    private schema: SchemaType;

    constructor() {
        this.config = config;
        const {base, table} = config;
        this.config.userPassword = process.env.CH_PASSWORD!;
        this.schema = new Schema(base, table);
    }

    private makeHttpRequestToClickHouse = async (requestData: IRequestData) => {
        const {port, userName, userPassword} = this.config;
        const {method, host, query} = requestData;
        return await superagent(method, `https://${host}:${port}`)
            .set({'X-ClickHouse-User': userName, 'X-ClickHouse-Key': userPassword})
            .send(query);
    }

    private withRetry = async (func: Function, params: IRequestData) => {
        const shuffledHosts = this.config.hosts.sort(() => Math.random() - 0.5);
        let lastError;
        for (let host of shuffledHosts) {
            try {
                const response = await func({...params, host: host});
                return response.status;
            } catch(error) {
                process.stderr.write(`${host} response with ${error}: ${error.response?.text}\n`);
                lastError = error
            }
        }
        return new Error(lastError);
    }

    public insertLogs = (logs: IData) => {
        const query = this.schema.insert(logs);
        const requestData: IRequestData = {
            method: HttpMethod.POST,
            query: query,
        }

        return this.withRetry(this.makeHttpRequestToClickHouse, requestData);
    }
}

export type ClickhouseClientType = ReturnType <() => ClickhouseClient>;
