import path from 'path';
import {
    performOAuthGetRequest,
    performOAuthPutRequest,
    performOAuthPostRequest,
} from "../../lib/helpers";
import {
    SandboxTaskQuery,
    SandboxObjectList,
    SandboxTaskAction,
    SandboxOwner,
    SandboxTaskType,
    SandboxTaskPriorityClass,
    SandboxTaskPrioritySubClass,
} from './types'
import TASK_TEMPLATE from './taskTemplate.json';

const DEFAULT_LIMIT = 100;
const API_URL = 'https://sandbox.yandex-team.ru/api/v1.0';
const INTERFACE_URL = 'https://sandbox.yandex-team.ru';
const CHANGE_STATUS_COMMENT = 'Status changed by neokrush webhook';

export class Sandbox {
    private static _instance: Sandbox;
    private _token: string;

    private constructor() {
        if (!Sandbox._instance) {
            Sandbox._instance = this;
        }
        return Sandbox._instance;
    }

    public static get instance(): Sandbox {
        return this._instance || (this._instance = new this());
    }

    public setToken = (token: string) => {
        this._token = token;
    };

    public async createTask<InputParams extends Record<string, unknown>>(
        type: SandboxTaskType,
        inputParams: InputParams,
        description?: string
    ): Promise<any> {
        const customFields = Object.entries(inputParams).map(([key, value]) => ({
            name: key,
            value,
        }));

        const requestBody = {
            ...TASK_TEMPLATE,
            custom_fields: customFields,
            description,
            owner: SandboxOwner.PARTNER_FRONT,
            priority: {
                class: SandboxTaskPriorityClass.SERVICE,
                subclass: SandboxTaskPrioritySubClass.HIGH,
            },
            type,
        };

        return await performOAuthPostRequest(
            `${API_URL}/task`,
            requestBody,
            this._token,
        );
    }

    public async getTasks(query: SandboxTaskQuery): Promise<SandboxObjectList> {
        return await performOAuthGetRequest(
            `${API_URL}/task?${this.makeGetTaskQuery(query)}`,
            (body) => {
                return body;
            },
            this._token,
        );
    }

    public getSandboxTaskUrl(taskId: number): string {
        return `${INTERFACE_URL}/task/${taskId}/view`;
    }

    public async startTasksByIds(ids: number[]): Promise<any> {
        return this.changeTasksStatusByIds(ids, SandboxTaskAction.START);
    }

    public async stopTasksByIds(ids: number[]): Promise<any> {
        return this.changeTasksStatusByIds(ids, SandboxTaskAction.STOP);
    }

    private async changeTasksStatusByIds(ids: number[], status: SandboxTaskAction): Promise<any> {
        return await performOAuthPutRequest(`${API_URL}/batch/tasks/${status}`,
            {id: ids, comment: CHANGE_STATUS_COMMENT},
            this._token,
        );
    }

    private makeGetTaskQuery(query: SandboxTaskQuery): string {
        let result = '';
        result += 'limit=' + Math.min(query.limit || DEFAULT_LIMIT, DEFAULT_LIMIT);
        result += '&owner=' + (query.owner || SandboxOwner.PARTNER_FRONT);
        if (query.descriptionRe) {
            result += `&desc_re=${(query.descriptionRe)}`;
        }
        if (query.status) {
            result += `&status=${query.status.join(',')}`;
        }
        if (query.type) {
            result += `&type=${query.type.join(',')}`;
        }
        return result;
    }
}
