import {
    performOAuthGetRequest,
    performOAuthPostRequest,
    performOAuthPatchRequest,
} from '../../lib/helpers';

import {Sprint, Ticket, TicketLink} from './types';

const STARTREK_URL = 'https://st-api.yandex-team.ru/v2';

let TOKEN = '';

export function setToken(token: string): void {
    TOKEN = token;
}

function getToken(): string {
    return TOKEN;
}

function parseTicket(body: any): Ticket {
    if (!body) {
        throw new Error('Startrek getTicket() response body is empty!');
    }

    if (!body.key) {
        throw new Error('Startrek getTicket() response has not essential ticket data!');
    }

    const ticket: Ticket = {
        id: body.key,
        self: body.self,
        summary: body.summary,
        createdBy: body.createdBy,
        assignee: body.assignee,
        status: body.status,
        storyPoints: body.storyPoints,
        originalStoryPoints: body.originalStoryPoints,
        tags: body.tags || [],
        start: body.start,
        end: body.end,
        sprint: body.sprint && body.sprint[0],
        currentStatusStartTime: body.statusStartTime,
        isFinished: Boolean(body.resolution),
        isError: false
    };

    // документация API Стартрека
    // https://wiki.yandex-team.ru/tracker/api/issues/get/
    return ticket;
}

function parseTicketDescription(body:any):string | void {
    if (!body) {
        throw new Error('Startrek getTicketLinks() response body is empty!');
    }

    return body.description || '';
}

function parseTicketLinks(body: any): TicketLink[] {
    if (!body) {
        throw new Error('Startrek getTicketLinks() response body is empty!');
    }

    if (!Array.isArray(body)) {
        throw new Error('Startrek getTicketLinks() response is not an array!');
    }

    // документация API Стартрека
    // https://wiki.yandex-team.ru/tracker/api/issues/links/list/
    return body as TicketLink[];
}

function parseSprint(body: any): Sprint {
    if (body === undefined || body === null) {
        throw new Error('Startrek getSprint() response body is empty!');
    }

    return {
        id: body.id,
        startDate: body.startDate,
        endDate: body.endDate,
    } as Sprint;
}

export async function getTicket(ticketId: string): Promise<Ticket | void> {
    const url = `${STARTREK_URL}/issues/${ticketId}`;

    return await performOAuthGetRequest(url, parseTicket, getToken())
        .catch((error) => {
            console.error(`Failed to get work breakdown structure: ${ticketId} - ${error}`);
        });
}

export async function getTicketDescription(ticketId: string): Promise<string | void> {
    const url = `${STARTREK_URL}/issues/${ticketId}`;

    return await performOAuthGetRequest(url, parseTicketDescription, getToken());
}

export async function getSprint(sprintId: string): Promise<Sprint | void> {
    const url = `${STARTREK_URL}/sprints/${sprintId}`;

    return await performOAuthGetRequest(url, parseSprint, getToken());
}

export async function getTicketLinks(ticketId: string): Promise<TicketLink[] | void> {
    const url = `${STARTREK_URL}/issues/${ticketId}/links`;

    return await performOAuthGetRequest(url, parseTicketLinks, getToken());
}

export async function createComment(ticketId: string, message: string): Promise<any> {
    const url = `${STARTREK_URL}/issues/${ticketId}/comments`;

    const requestBody = {
        text: message,
    };

    return await performOAuthPostRequest(url, requestBody, getToken());
}

export async function updateDescription(ticketId: string, description: string): Promise<any> {
    const url = `${STARTREK_URL}/issues/${ticketId}`;

    const requestBody = {
        description,
    };

    return await performOAuthPatchRequest(url, requestBody, getToken());
}
