import {getSprint, getTicket, getTicketLinks} from '../index';
import {flatten, intersect} from '../../../lib/helpers';

import {Sprint, Ticket, TicketLink, TicketLinkDirection, TicketLinkType} from '../types';
import {ErrorTask, TicketNodeWithChildrenIds as TicketNode, TimelineTicket} from './types';

// функции, связанные с построением дерева из задач
export async function getTimelineTasks(parentTicketId: string, nest = 0): Promise<TimelineTicket[]> {
    const parentTicket = await getTicket(parentTicketId);

    if (parentTicket) {
        const parentTicketNode = await getTicketLinks(parentTicketId)
            .then((parentTicketLinks) => processStartrekResponse(
                parentTicket as Ticket, parentTicketLinks as TicketLink[], nest,
            ));

        if (parentTicketNode.children.length > 0) {
            const childTasks = await mapChildTimelineTasks(parentTicketNode.children, nest + 1);

            const flattenChildrenTasks = flatten(childTasks);

            return [parentTicketNode, ...flattenChildrenTasks];
        }

        return [parentTicketNode];
    }

    return Promise.resolve([{
        id: parentTicketId,
        isError: true,
        start: '',
        end: '',
        nest,
    }])


}

async function mapChildTimelineTasks(childIds: string[], nest: number) {
    return Promise.all(childIds.map((id) => {
        return getTimelineTasks(id, nest);
    }));
}

function processStartrekResponse(parentTicket: Ticket, parentTicketLinks: TicketLink[], nest: number): TicketNode {
    let childrenTicketIds: string[] = [];
    let grandparent = null;

    if (parentTicketLinks.length > 0) {
        const subtasks = parentTicketLinks.filter((link) => link.type.id === TicketLinkType.Subtask && link.direction === TicketLinkDirection.Outward);
        [grandparent] = parentTicketLinks.filter((link) => link.type.id === TicketLinkType.Subtask && link.direction === TicketLinkDirection.Inward);
        childrenTicketIds = subtasks.reduce((acc, item) => {
            return acc.concat(item.object.key)
        }, []  as string[]);
    }

    return {
        ...parentTicket,
        children: childrenTicketIds,
        parentId: nest === 0 ? null : (grandparent && grandparent.object.key),
        nest,
    }
}

export function filterTimelineTasks(tasks: TimelineTicket[], filterTags: string[]) {
    return tasks.filter((task) => {
        if (task.isError) {
            return true;
        }

        const intersection = intersect(task.tags, filterTags);

        return intersection.length === 0;
    });
}

export function separateErrorTasks(tasks: TimelineTicket[]): {errorTasks: ErrorTask[], validTasks: TicketNode[]} {
    const errorTasks: ErrorTask[] = [];

    const filterFunction = (task: TimelineTicket): task is TicketNode => {
        if (task.isError) {
            errorTasks.push(task);
            return false;
        }
        return true;
    };

    const validTasks: TicketNode[] = tasks.filter(filterFunction);

    return {
        errorTasks,
        validTasks,
    };
}

export async function enhanceTimelineTasksWithDates(tasks: TimelineTicket[]): Promise<TimelineTicket[]> {
    let tasksWithoutDates: TicketNode[] = [];

    const {errorTasks, validTasks} = separateErrorTasks(tasks);

    const sprintIds = validTasks.filter((task) => task.sprint).map((taskWithSprint) => taskWithSprint.sprint.id);
    const sprints: {[key: string]: Sprint} = await getSprints(sprintIds);

    // отделяем те тикеты, где явно указаны даты, или есть спринт
    const tasksWithDates = validTasks.map((task) => {
        const areDatesDefined = Boolean(task.start && task.end);

        if (areDatesDefined) {
            return task;
        }

        if (!areDatesDefined && task.parentId === null) {
            throw Error(`Для корневого тикета ${task.id} не определен временной промежуток`);
        }

        if (!areDatesDefined && task.sprint) {
            const {id: sprintId} = task.sprint;

            return {
                ...task,
                start: sprints[sprintId].startDate,
                end: sprints[sprintId].endDate,
            }
        }

        tasksWithoutDates.push(task);
        return null;
    });

    const cleanTasksWithDates: TicketNode[] = tasksWithDates.filter((item): item is TicketNode => item !== null);

    // разбираем тикеты без дат
    // если у тикета не проставлены даты явно и нет спринта - ищем его родителя в тикетах с датами
    // если нашли - проставляем текущем даты и убираем его из списка "без дат"
    // если нет - пропускаем, так как у его родителя тоже не проставлены даты, он обработается на следующих итерациях
    while (tasksWithoutDates.length) {
        const datedTaskIds: string[] = [];
        for (let i = 0; i < tasksWithoutDates.length; i++) {
            const currentItem = tasksWithoutDates[i];

            const parentItem = cleanTasksWithDates.find((task) => task.id === currentItem.parentId);

            if (parentItem) {
                cleanTasksWithDates.push({
                    ...currentItem,
                    start: parentItem.start,
                    end: parentItem.end,
                });

                datedTaskIds.push(currentItem.id);
            }
        }

        tasksWithoutDates = tasksWithoutDates.filter((task) => !datedTaskIds.includes(task.id))
    }

    const rootTask = cleanTasksWithDates.find((task) => task.parentId === null);

    if (!rootTask) {
        throw Error(`Не получилось найти корневой тикет в массиве`);
    }

    // тикетам, данные по которым не получилось получить, проставляем даты корневого тикета
    const errorTasksWithDates: ErrorTask[] = errorTasks.map((task) => ({
        ...task,
        start: rootTask.start,
        end: rootTask.end,
    }));

    return [...cleanTasksWithDates, ...errorTasksWithDates];
}

async function getSprints(sprintsIds: string[]) {
    const sprintsArray: (Sprint | void)[] = await Promise.all(sprintsIds.map((id) => {
        return getSprint(id);
    }));

    return sprintsArray.reduce((acc, sprint) => {
        if (sprint) {
            return {
                ...acc,
                [sprint.id]: sprint,
            }
        }

        return acc;
    }, {})
}
