import {getTicket, getTicketLinks} from '../index';
import {intersect} from '../../../lib/helpers';
import {getLineForNode} from '../../../lib/formatters/nodeToMarkdown';

import {FilteredTickets} from '../../../interface/webhooks/service/wbs/types';
import {Ticket, TicketLink, TicketLinkType, TicketLinkDirection, isTicketNode} from '../types';
import {TicketNodeWithChildren as TicketNode,  WbsTicket} from './types';

// функции, связанные с построением дерева из задач
export async function getWbs(parentTicketId: string, filterTags: string[]): Promise<[TicketNode, FilteredTickets]> {
    const rawTreeStructure: WbsTicket = await buildTicketsTree(parentTicketId, 0);

    if (rawTreeStructure.isError) {
        throw new Error('Ошибка при получении проектного тикета');
    }

    if (filterTags.length > 0) {
        // строим новое дерево, в котором нет тикетов с указанными тегами, отфильтрованные тикеты возращаем отдельной структурой
        const [newTreeStructure, filteredNodes] = filterTreeWithTags(rawTreeStructure, filterTags, {});

        if (newTreeStructure === null) {
            throw new Error('Все элементы структуры были отфильтрованы');
        }

        if (!isTicketNode(newTreeStructure)) {
            // корень структуры не может быть ErrorTicket
            throw new Error('Ошибка при получении проектного тикета');
        }

        // шаблон структуры отфильтрованных тикетов (тег - пустой массив)
        const defaultFilteredNodes: FilteredTickets = filterTags.reduce((result, tag) => {
            return {
                ...result,
                [tag]: [],
            }
        }, {});

        const resultedFilteredNodes = {
            ...defaultFilteredNodes,
            ...filteredNodes,
        };

        return [newTreeStructure, resultedFilteredNodes];
    }

    return [rawTreeStructure, {}];
}

export async function buildTicketsTree(parentTicketId: string, nest: number): Promise<WbsTicket> {
    const parentTicket = await getTicket(parentTicketId);

    if (parentTicket) {
        return await getTicketLinks(parentTicketId).then((
            parentTicketLinks,
        ) => processStartrekResponse(parentTicket as Ticket, parentTicketLinks as TicketLink[], nest));
    }

    return Promise.resolve({
        id: parentTicketId,
        isError: true,
    })

}

async function processStartrekResponse(
    parentTicket: Ticket, parentTicketLinks: TicketLink[], nest: number,
): Promise<TicketNode> {
    let childrenTickets: WbsTicket[] = [];
    let grandparent = null;

    if (parentTicketLinks.length > 0) {
        const subtasks = parentTicketLinks
            .filter((link) => link.type.id === TicketLinkType.Subtask && link.direction === TicketLinkDirection.Outward);
        [grandparent] = parentTicketLinks
            .filter((link) => link.type.id === TicketLinkType.Subtask && link.direction === TicketLinkDirection.Inward);

        childrenTickets = await mapParentTicketLinks(subtasks, nest + 1);
    }

    return {
        ...parentTicket,
        children: childrenTickets,
        parentId: grandparent && grandparent.object.key,
        nest,
    }
}

async function mapParentTicketLinks(links: TicketLink[], nest: number) {
    return Promise.all(links.map((link) => {
        return buildTicketsTree(link.object.key, nest);
    }));
}

// функция фильтрации дерева задач
export function filterTreeWithTags(parentNode: WbsTicket, filterTags: string[], filteredNodes: FilteredTickets): [WbsTicket | null, FilteredTickets] {
    if (parentNode.isError) {
        return [parentNode, filteredNodes];
    }

    const intersection = intersect(parentNode.tags, filterTags);

    if (intersection.length > 0) {
        intersection.forEach((tag) => {
            if (filteredNodes[tag]) {
                filteredNodes[tag].push(parentNode);
            } else {
                filteredNodes[tag] = [parentNode];
            }
        });

        return [null, filteredNodes]
    }

    if (parentNode.children.length) {
        const childrenTickets = parentNode.children.map((child: TicketNode) => {
            const [result, filtered] = filterTreeWithTags(child, filterTags, filteredNodes);

            filteredNodes = filtered;

            return result;
        });

        parentNode.children = childrenTickets.filter((item) => item !== null) as TicketNode[];
    }


    return [parentNode, filteredNodes];
}

export function getFilteredTicketsDescription(filteredTickets: FilteredTickets, rootTicketId: string): string {
    const result = Object.keys(filteredTickets).reduce((accDescription: string, tag: string) => {
        const ticketsByTag = filteredTickets[tag];

        if (!ticketsByTag.length) {
            return accDescription.concat(``);
        }

        const ticketsByTagDescription = ticketsByTag.reduce((accDescription: string, ticket: TicketNode) => {
            const ticketDescription = `${getLineForNode(ticket)}\n`;
            return accDescription.concat(ticketDescription);
        }, '');

        return accDescription.concat(`=====Отфильтрованные по тегу %%${tag}%%\n${ticketsByTagDescription}`);
    }, '');

    return `---------\n${result}`
}
