import * as React from 'react';
import superagent from 'superagent';
import {compose} from '@bem-react/core';

import {Spin as SpinDesktop, withSizeM, withViewDefault} from '@yandex-lego/components/Spin/desktop';

import {getQueryParams} from '../../../lib/location/getQueryParams';
import {FormInputGroup} from '../components/FormGroup';
import {Form} from '../components/Form';
import {Timeline} from './Timeline';
import {Page} from '../components/PageSwitcher';

import {RawTimelineRequestBody as TimelineRequestBody, TimelineData} from '../../../interface/webhooks/service/timeline/types';
import {PageProps, RequestStatus} from '../types';

import styles from './styles.css';

const MAKE_TIMELINE_URL = `${HOST}/timeline/make`;

const Spin = compose(withSizeM, withViewDefault)(SpinDesktop);

const DEFAULT_TIMELINE_DATA: TimelineData = {
    tasksGroups: {
        groups: [],
        groupsWithNest: []
    },
    tasksItems: [],
};

export const TimelineForm = (props: PageProps) => {
    const {location, history} = props;
    const queryParams = getQueryParams(location);

    const [ticket, setTicket] = React.useState(typeof queryParams.ticket === 'string' ? queryParams.ticket : '',);
    const [errorMessage, setErrorMessage] = React.useState('');
    const [filters, setFilters] = React.useState('');
    const [timelineData, setTimelineData] = React.useState(DEFAULT_TIMELINE_DATA);
    const [requestStatus, setRequestStatus] = React.useState(RequestStatus.notStarted);

    const onButtonClickHandler = () => {
        const requestBody: TimelineRequestBody = {
            ticketId: ticket,
            filterTags: filters,
        };

        setRequestStatus(RequestStatus.pending);

        return superagent
            .post(MAKE_TIMELINE_URL)
            .send(requestBody)
            .then((result) => {
                if (result.status === 200 && !result.body?.error) {
                    setTimelineData(result.body);
                    setRequestStatus(RequestStatus.success);
                } else {
                    setErrorMessage(result.body.error);
                    setRequestStatus(RequestStatus.fail);
                }
            })
            .catch((error) => {
                console.error(error);
                setRequestStatus(RequestStatus.fail);
            })
    };

    const onTicketChangeHandler = (value: string) => {
        setTicket(value);
        setRequestStatus(RequestStatus.notStarted);
        history.replace(`${Page.timeline}?ticket=${value}`);
    };

    const onFiltersChangeHandler = (value: string) => {
        setFilters(value);
        setRequestStatus(RequestStatus.notStarted);
    };

    return (
        <div className={styles.page}>
            <div className={styles.wrapper}>
                <Form
                    headline="Построение графика проекта"
                    onButtonClickHandler={onButtonClickHandler}
                    buttonText="Построить график"
                >
                    <FormInputGroup
                        label="Проектный тикет"
                        placeholder="PPD-123"
                        hint="Номер проектного тикета в очереди PPD"
                        value={ticket}
                        onChange={onTicketChangeHandler}
                        required={true}
                    />
                    <FormInputGroup
                        label="Фильтры"
                        placeholder="tag1, tag2, tag3"
                        hint="Теги тикета, по которым нужно исключать его из структуры"
                        value={filters}
                        onChange={onFiltersChangeHandler}
                    />
                </Form>
                {
                    requestStatus === RequestStatus.pending && (
                        <div className={styles.spin}>
                            <Spin progress={true} view="default" size="m" />
                        </div>
                    )
                }
                {
                    requestStatus === RequestStatus.fail && (
                        <>
                            <span>Упс! Что-то пошло не так...</span>
                            <div className={styles.errorMessage}>{errorMessage}</div>
                        </>
                    )
                }
            </div>
            {
                requestStatus === RequestStatus.success && timelineData.tasksItems.length > 0
                    ? (
                        <div className={styles.chart}>
                            <Timeline groups={timelineData.tasksGroups} items={timelineData.tasksItems}/>
                        </div>
                    )
                    : null
            }
        </div>
    )
};
