import * as React from 'react';
import superagent from 'superagent';
import qs from 'qs';
import htmlParser from 'html-react-parser';
import {History} from 'history';

import {compose} from '@bem-react/core';
import {Link as LinkDesktop, withViewDefault as withLinkViewDefault} from '@yandex-lego/components/Link/desktop';
import {
    Spin as SpinDesktop,
    withSizeM as withSpinSizeM,
    withViewDefault as withSpinViewDefault
} from '@yandex-lego/components/Spin/desktop'
import {Textinput, withSizeM, withViewDefault} from '@yandex-lego/components/Textinput/desktop'
import {Button} from '@yandex-lego/components/Button/desktop/bundle';
import {Checkbox} from '@yandex-lego/components/Checkbox/desktop/bundle'
import {Select} from '@yandex-lego/components/Select/desktop/bundle';

import {getQueryParams} from '../../../lib/location/getQueryParams';
import {FormInputGroup} from '../components/FormGroup';
import {Form} from '../components/Form';
import {Page} from '../components/PageSwitcher';

import {
    BudgetPreset,
    RawWbsRequestBody as WbsRequestBody,
    WbsErrorFields
} from '../../../interface/webhooks/service/wbs/types';
import {RequestStatus, PageProps} from '../types';

import styles from './styles.css';

const Link = compose(withLinkViewDefault)(LinkDesktop);
const Spin = compose(withSpinSizeM, withSpinViewDefault)(SpinDesktop);
const TextInput = compose(withViewDefault, withSizeM)(Textinput);

interface FieldState {
    value: string;
    error?: string;
}

interface WbsState {
    projectTicket: FieldState;
    wbsTicket: FieldState;
    filters: FieldState;
    preset: FieldState;
    requestStatus: RequestStatus;
    description: string;
    isBudget: boolean;
    shouldWriteToStartrek: boolean;
    generalError: string;
}

const MAKE_WBS_URL = `${HOST}/wbs/make`;
const STARTREK_HOST = 'https://st.yandex-team.ru/';

export class Wbs extends React.PureComponent {
    public state: WbsState;

    private history: History;

    constructor(props: PageProps) {
        super(props);

        const {location, history} = props;
        const queryParams = getQueryParams(location);

        this.history = history;

        this.state = {
            projectTicket: {
                value: typeof queryParams.ticket === 'string' ? queryParams.ticket : '',
                error: '',
            },
            wbsTicket: {
                value: '',
                error: '',
            },
            filters: {
                value: 'excludeFromBudget',
                error: '',
            },
            preset: {
                value: BudgetPreset.ppd,
                error: '',
            },
            isBudget: true,
            shouldWriteToStartrek: false,
            requestStatus: RequestStatus.notStarted,
            description: '',
            generalError: '',
        }
    }

    private onProjectTicketIdChangeHandler = (value: string) => {
        this.setState((state: WbsState): WbsState => ({
            ...state,
            projectTicket: {
                value,
                error: '',
            },
            requestStatus: RequestStatus.notStarted,
            description: '',
        }));
        this.history.replace(`${Page.wbs}?ticket=${value}`);
    };

    private onWbsTicketIdChangeHandler = (event: React.SyntheticEvent<HTMLInputElement>) => {
        const {value} = event.currentTarget;
        this.setState((state: WbsState): WbsState => ({
            ...state,
            wbsTicket: {
                value,
                error: '',
            },
            requestStatus: RequestStatus.notStarted,
            description: '',
        }));
    };

    private onFiltersChangeHandler = (value: string) => {
        this.setState((state: WbsState): WbsState => ({
            ...state,
            filters: {
                value,
                error: '',
            },
            requestStatus: RequestStatus.notStarted,
            description: '',
        }));
    };

    private onQueuesChangeHandler = (event: any) => {
        this.setState((state: WbsState): WbsState => ({
            ...state,
            preset: {
                value: event.target.value,
                error: '',
            },
            requestStatus: RequestStatus.notStarted,
            description: '',
        }));
    };

    private onBudgetCheckHandler = () => {
        this.setState((state: WbsState): WbsState => ({
            ...state,
            requestStatus: RequestStatus.notStarted,
            description: '',
            isBudget: !state.isBudget,
        }));
    };

    private onWbsTicketCheckHandler = () => {
        this.setState((state: WbsState): WbsState => ({
            ...state,
            requestStatus: RequestStatus.notStarted,
            description: '',
            shouldWriteToStartrek: !state.shouldWriteToStartrek,
        }));
    };

    private parseWbsError = (error: WbsErrorFields) => {
        if (error.ticketId) {
            this.setState((state: WbsState): WbsState => ({
                ...state,
                projectTicket: {
                    value: state.projectTicket.value,
                    error: error.ticketId,
                },
                requestStatus: RequestStatus.fail,
            }));
        }
        if (error.outputTicketId) {
            this.setState((state: WbsState): WbsState => ({
                ...state,
                wbsTicket: {
                    value: state.wbsTicket.value,
                    error: error.outputTicketId,
                },
                requestStatus: RequestStatus.fail,
            }));
        }
        if (error.general) {
            this.setState((state: WbsState): WbsState => ({
                ...state,
                generalError: error.general,
                requestStatus: RequestStatus.fail,
            }) as WbsState);
        }
    };

    private onButtonClickHandler = () => {
        const {projectTicket, wbsTicket, filters, preset, isBudget, shouldWriteToStartrek} = this.state;

        const requestBody: WbsRequestBody = {
            ticketId: projectTicket.value.toUpperCase(),
            filterTags: filters.value,
            outputTicketId: shouldWriteToStartrek ? wbsTicket.value.toUpperCase() : undefined,
            budgetPreset: isBudget ? preset.value : undefined,
        };

        this.setState((state: WbsState): WbsState => ({
            ...state,
            requestStatus: RequestStatus.pending,
        }));

        return superagent
            .post(MAKE_WBS_URL)
            .send(requestBody)
            .then((result) => {
                if (result.status === 200 && !result.body?.error) {
                    this.setState((state: WbsState): WbsState => ({
                        ...state,
                        projectTicket: {
                            value: state.projectTicket.value,
                            error: '',
                        },
                        wbsTicket: {
                            value: state.wbsTicket.value,
                            error: '',
                        },
                        filters: {
                            value: state.filters.value,
                            error: '',
                        },
                        preset: {
                            value: state.preset.value,
                            error: '',
                        },
                        generalError: '',
                        requestStatus: RequestStatus.success,
                        description: result.text.length > 2 ? result.text : ''
                    }));
                } else {
                    this.parseWbsError(result.body.error);
                }
            })
            .catch((agentError) => {
                this.setState((state: WbsState): WbsState => ({
                    ...state,
                    requestStatus: RequestStatus.fail,
                }));
                if (agentError?.response) {
                    console.error(agentError?.response);
                }
                console.error(agentError);
            })
    };

    render(): React.ReactElement {
        const {
            projectTicket,
            wbsTicket,
            filters,
            preset,
            requestStatus,
            description,
            isBudget,
            shouldWriteToStartrek,
        } = this.state;

        return (
            <div className={styles.page}>
                <div className={styles.wrapper}>
                    <Form
                        headline="Построение Иерархической Структуры Работ (ИСР) проекта"
                        onButtonClickHandler={this.onButtonClickHandler}
                        buttonText="Построить ИСР"
                    >
                        <FormInputGroup
                            label="Проектный тикет"
                            placeholder="PPD-123"
                            hint={projectTicket.error || 'Номер проектного/группового тикета'}
                            state={projectTicket.error ? 'error' : undefined}
                            value={projectTicket.value}
                            onChange={this.onProjectTicketIdChangeHandler}
                            required={true}
                        />
                        <FormInputGroup
                            label="Фильтры"
                            placeholder="tag1, tag2, tag3"
                            hint="Теги тикета, по которым нужно исключать его из расчёта бюджета"
                            value={filters.value}
                            onChange={this.onFiltersChangeHandler}
                        />
                        <div className={styles.additionalSettings}>
                            <Checkbox
                                label="Рассчитать бюджет"
                                size="m"
                                view="default"
                                onChange={this.onBudgetCheckHandler}
                                checked={isBudget}
                                className={styles.checkbox}
                            />
                            <Select
                                size="m"
                                view="default"
                                options={[
                                    {value: BudgetPreset.ppd, content: BudgetPreset.ppd},
                                    {value: BudgetPreset.all, content: BudgetPreset.all}
                                ]}
                                disabled={!isBudget}
                                onChange={this.onQueuesChangeHandler}
                                value={preset.value}
                            />
                            <Checkbox
                                label="Записать в тикет"
                                size="m"
                                view="default"
                                onChange={this.onWbsTicketCheckHandler}
                                checked={shouldWriteToStartrek}
                                className={styles.checkbox}
                            />
                            <TextInput
                                className={styles.wbsTicket}
                                size="m"
                                view="default"
                                value={wbsTicket.value}
                                placeholder="Ключ трекера"
                                hint={wbsTicket.error}
                                state={wbsTicket.error ? 'error' : undefined}
                                onChange={this.onWbsTicketIdChangeHandler}
                                disabled={!shouldWriteToStartrek}
                            />
                        </div>
                    </Form>
                    {
                        requestStatus === RequestStatus.pending && (
                            <div className={styles.spin}>
                                <Spin progress={true} view="default" size="m" />
                            </div>
                        )
                    }
                    {
                        description && requestStatus === RequestStatus.success && this.renderSuccessStatusHtml(description)
                    }
                    {
                        requestStatus === RequestStatus.fail && (
                            <>
                                <span className={styles.oops}>Упс! Что-то пошло не так...</span>
                                <div>{this.state.generalError}</div>
                            </>
                        )
                    }
                </div>
            </div>
        );
    }

    private renderSuccessStatusLink = (ticketId: string) => {
        return (
            <>
                <span>Готово! Перейдите по ссылке в </span>
                <Link
                    view="default"
                    href={`https://st.yandex-team.ru/${ticketId}`}
                    target="_blank"
                >
                    тикет
                </Link>
            </>
        )
    };

    private replaceInitialSpaces = (text: string) => {
        const dotIndex = text.indexOf('.') - 1;
        const starIndex = text.indexOf('*') - 1;

        const index = dotIndex < 1 ? starIndex : dotIndex;

        if (index < 1) {
            return text;
        }
        else {
            let prefix = '';
            const space = '\u00A0';

            for (let i = 0; i < index - 1; i++) {
                prefix += space
            };
            return `${prefix}${text}`;
        }
    };

    private formatMarkdownTreeForHTML = (text: string) => {
        return (
            text
                // расставляем заголовки
                .replace(/(^|===|Показатели бюджета)(.)+\n/g, '<h3>$&</h3>')
                // тикет заменяем на ссылку на него в Трекере
                .replace(/[a-zA-Z]{2,}-[0-9]+/g, `<a href='${STARTREK_HOST}$&' target=_blank>$&</a>`)
                // добавляем отступы
                .replace(/ +(\d+.)+[+ ]/g, this.replaceInitialSpaces)
                .replace(/ +\*/g, this.replaceInitialSpaces)
                // стили
                .replace(/ (\d+[.])+([+]? )/g, `<span class="${styles.lightgray}">$&</span>`)
                .replace(/ +\* /g, `<span class="${styles.lightgray}">$&</span>`)
                .replace(/\*\*.*\*\*/g, `<span class="${styles.summaryTask}">$&</span>`)
                .replace(/(--.*--|\[Закрыт\])/g, `<span class="${styles.closedTask}">$&</span>`)
                .replace(/\n/g, '<br />')
        );
    };

    private renderSuccessStatusHtml = (tree: string) => {
        const onCopyHandler = () => {
            let temp = document.createElement('textarea');
            temp.value = tree;

            document.body.appendChild(temp);
            temp.select();
            document.execCommand('copy');
            document.body.removeChild(temp);
        };

        const {shouldWriteToStartrek, wbsTicket} = this.state;
        const htmlTree = this.formatMarkdownTreeForHTML(tree);

        return (
            <div className={styles.result}>
                {shouldWriteToStartrek && this.renderSuccessStatusLink(wbsTicket.value)}
                <div>Результат</div>
                <div className={styles.markdown}>
                    {htmlParser(htmlTree)}
                </div>
                <div className={styles.copyTreeButtonWrapper}>
                    <Button
                        view="default"
                        size="m"
                        theme="action"
                        type="submit"
                        className={styles.button}
                        onClick={onCopyHandler}
                    >
                        Cкопировать в буфер
                    </Button>
                </div>
            </div>
        )
    };
}

