import {apiCall, apiSearch} from '../../startrek';
import {loadParentsRec} from '../../lib/loadParentsRec';
import {QuarterBlock} from './QuarterConfigurator.types';

export const detectChildren = (store: Record<string, any> = {}) => {
    const issues = Object.values(store);

    for (const issue of issues) {
        if (issue.parent && store[issue.parent.key]) {
            store[issue.parent.key].children = store[issue.parent.key].children ?
                [...store[issue.parent.key].children, issue.key] :
                [issue.key];
        }
    }

    return store;
}

export const computeStoryPoints = (store: Record<string, any> = {}) => {
    const issues = Object.values(store);

    for (const issue of issues) {
        if (!issue.children) {
            issue.computedStoryPoints = issue.originalStoryPoints || issue.storyPoints || 0;

            if (issue.parent && store[issue.parent.key]) {
                let parent = store[issue.parent.key];

                while (parent) {
                    store[parent.key].computedStoryPoints = (store[parent.key].computedStoryPoints || 0) +
                        (issue.computedStoryPoints || 0);

                    parent = parent.parent && store[parent.parent.key];
                }
            }
        }
    }

    return store;
}

/**
 * Добавляет название борды по ее уникальному ключу, сделано для того, чтобы, если борда была
 * переименована информация оставалась актуальной
 * @param config
 */
export const setBoard = async (config: {boardId?: string}) => {
    if (config.boardId) {
        const uri = `boards/${config.boardId}`;
        const board = await apiCall(uri);

        return {...config, boardName: board.name};
    }

    return config;
}

/**
 * Добавляет информацию по спринтам по сохраненным в задаче айдишникам
 * @param config
 */
export const setSprints = async (config: {boardId?: string, sprints: Array<Record<string, any>>}) => {
    if (config.boardId) {
        const uri = `boards/${config.boardId}/sprints`;
        const sprints = Object.fromEntries(
            (await apiCall(uri)).map((sprint: Record<string, any>) => [sprint.id, sprint])
        );

        return {...config, sprints: config.sprints.map(sprint => ({...sprint, ...sprints[sprint.id]}))};
    }

    return config;
}

/**
 * Загружает и добавляет информацию по задачам из продуктов с метчингом на используемые в рамках квартала спринты
 * @param config
 */
export const setIssues = async (config: {
    sprints: Array<Record<string, any>>;
    boardName?: string,
    blocks?: Array<QuarterBlock>;
    filter?: string;
}) => {
    if (config.boardName && config.blocks) {
        const sprintsQueryString = config.sprints.map(sprint => sprint.name).filter((sprintName) => sprintName)
            .map(sprintName => '"' + sprintName + '"')
            .join();

        if (!sprintsQueryString) {
            return {...config, issuesMap: {}};
        }

        const savedIssuesKeysString = config.blocks
            .map(block => block.products.map((product) => product.issueKey))
            .flat()
            .filter(issueId => typeof issueId === 'string')
            .join();

        const queryFilter = config.filter ? `AND (${config.filter})` : '';
        const querySuffix = savedIssuesKeysString ? `OR Key: ${savedIssuesKeysString}` : '';

        const query = `Boards: "${config.boardName}" AND Sprint: ${sprintsQueryString} ${queryFilter} ${querySuffix}`;

        const issues = await apiSearch(query, 600);
        const issuesMap = Object.fromEntries(issues.map(issue => [issue.key, issue]));


        const issuesKeys = Object.keys(issuesMap);

        await loadParentsRec(new Set(issuesKeys), issuesMap);

        return {...config, issuesMap};
    }

    return {...config, issuesMap: {}};
}

/**
 * Фильтр для сохраненния только необходимых параметров конфига в задаче
 * @param config
 */
export const saveConverter = (config: Record<string, any>) => {
    return {
        quarterName: config.quarterName,
        boardId: config.boardId,
        blocks: config.blocks,
        filter: config.filter,
        sprints: config.sprints.map((sprint: Record<string, any>) => ({
            id: sprint.id,
            description: sprint.description,
            totalCapacity: sprint.totalCapacity
        })),
    }
}

/**
 * Добавляет в задачи ссылки на дочерние задачи
 * @param config
 */
export const setChildren = (config: {issuesMap?: Record<string, any>}) => {
    const issuesMap = detectChildren(config.issuesMap);
    return {...config, issuesMap};
}

/**
 * Добавляет в задачи посчитанные накопленные значения story points
 * @param config
 */
export const setStoryPoints = (config: {issuesMap?: Record<string, any>}) => {
    const issuesMap = computeStoryPoints(config.issuesMap);
    return {...config, issuesMap};
}
