import React, {ChangeEvent, EventHandler, Reducer, useEffect, useReducer, useState} from 'react';
import {withNaming} from '@bem-react/classname';
import {Button} from '@yandex-lego/components/Button/desktop/bundle';
import {Textinput} from '@yandex-lego/components/Textinput/desktop/bundle';
import {Spin} from '@yandex-lego/components/Spin/desktop/bundle';

import {
    addBlockAction,
    addSprintAction, deleteBlockAction,
    deleteSprintAction,
    QUARTER_CONFIGURATOR_ACTION,
    QuarterConfiguratorAction, setBoardAction,
    setConfigAction,
    setFilterAction,
    setNameAction,
    setShouldNotSaveAction,
    setShouldNotUpdateAction, updateBlockInfoAction,
    updateSprintInfoAction,
} from './QuarterConfigurator.actions';
import {
    BoardSuggest,
    ExtendedIssue,
    QuarterBlock,
    QuarterSprint,
    QuarterTableSprintInfo,
} from './QuarterConfigurator.types';
import {QuarterTable, QuarterTableProps} from '../QuarterTable/QuarterTable';
import {QuarterTableRowProps} from '../QuarterTable/QuarterTableRow/QuarterTableRow';
import {ConfirmIcon} from '../Icons/ConfirmIcon';
import {EditIcon} from '../Icons/EditIcon';
import {QuarterConfiguratorBlock} from './QuarterConfiguratorBlock/QuarterConfiguratorBlock';
import {Suggest} from '../Suggest/Suggest';
import {QuarterTableCellEditableText} from '../QuarterTable/QuarterTableCellEditableText/QuarterTableCellEditableText';
import {QuarterTableCellSprint} from '../QuarterTable/QuarterTableCellSprintSelect/QuarterTableCellSprintSelect';
import {defaultQuarterConfig, prefixParamsStructure, suffixStructure} from './QuarterConfigurator.const';
import {SyncIcon} from '../Icons/SyncIcon';
import {DownloadIcon} from '../Icons/Download';

import './QuarterConfigurator.css';


const generateTableStructure = (sprintsNumber: number) => {
    const sprintsStructure: QuarterTableRowProps['structure'] = [{
        backgroundColor: 'rgb(255, 234, 158)',
        cells: Array.from({length: sprintsNumber}, () => ({
            type: 'default',
        })),
    }];

    return prefixParamsStructure.concat(sprintsStructure, suffixStructure);
};

interface QuarterConfiguratorProps {
    config: QuarterConfiguratorState | {};

    setConfig(nextConfig: Partial<QuarterConfiguratorState>, noMiddlewares?: boolean): void;

    updateConfig(nextConfig: QuarterConfiguratorState): void;

    updating: boolean;
    saveInProgress: boolean;
    className?: string;
}

export interface QuarterConfiguratorState {
    quarterName: string;
    boardId?: string;
    sprints: Array<QuarterSprint>;
    issuesMap: Record<string, any>;
    blocks: Array<QuarterBlock>;
    boardName?: string;
    shouldUpdateInfo?: boolean;
    shouldSaveInfo?: boolean;
    filter?: string;
}


const iconEdit = (className: string) => <EditIcon className={className} />;
const iconConfirm = (className: string) => <ConfirmIcon className={className} />;
const iconSync = (className: string) => <SyncIcon className={className} />;
const iconUpload = (className: string) => <DownloadIcon className={`${className} ${cn('upload-icon')}`} />;

const checkIsNotEmptyConfig = (config: QuarterConfiguratorProps['config']): config is QuarterConfiguratorState => {
    return Boolean((config as QuarterConfiguratorState).sprints);
};

const reducer: Reducer<QuarterConfiguratorProps['config'], QuarterConfiguratorAction> = (state, action) => {
    switch (action.type) {
        case QUARTER_CONFIGURATOR_ACTION.SET_CONFIG:
            return {
                ...action.payload,
                shouldSaveInfo: false,
            };
        case QUARTER_CONFIGURATOR_ACTION.SET_NAME:
            return {
                ...state,
                quarterName: action.payload,
                shouldSaveInfo: true,
            };
        case QUARTER_CONFIGURATOR_ACTION.SET_FILTER:
            return {
                ...state,
                filter: action.payload,
                shouldUpdateInfo: true,
                shouldSaveInfo: true,
            };
        case QUARTER_CONFIGURATOR_ACTION.SET_BOARD:
            return {
                ...state,
                boardName: action.payload.display,
                boardId: action.payload.id,
                shouldUpdateInfo: true,
                shouldSaveInfo: true,
            };
        case QUARTER_CONFIGURATOR_ACTION.UPDATE_SPRINT: {
            if (!checkIsNotEmptyConfig(state)) {
                return {...state};
            }

            const sprints = [...state.sprints];
            let updatedBlocks = state.blocks;

            if (Object.keys(action.payload.update).includes('description')) {
                updatedBlocks = updatedBlocks.map(block => ({
                    ...block,
                    products: block.products.map(product => {
                        const updatedSprints = {...product.sprints};
                        updatedSprints[action.payload.update.description!] =
                            {...updatedSprints[sprints[action.payload.index].description]};

                        return {
                            ...product,
                            sprints: updatedSprints,
                        };
                    }),
                }));
            }

            const shouldUpdateInfo = Object.keys(action.payload.update).includes('id');

            sprints[action.payload.index] = {...sprints[action.payload.index], ...action.payload.update};

            return {
                ...state,
                sprints,
                blocks: updatedBlocks,
                shouldUpdateInfo,
                shouldSaveInfo: true,
            };
        }
        case QUARTER_CONFIGURATOR_ACTION.ADD_SPRINT:
            if (!checkIsNotEmptyConfig(state)) {
                return {...state};
            }

            const sprints = [...state.sprints].concat({
                description: 'new sprint',
                totalCapacity: 0,
            });

            return {
                ...state,
                sprints,
                shouldSaveInfo: true,
            };
        case QUARTER_CONFIGURATOR_ACTION.UPDATE_BLOCK_INFO: {
            if (!checkIsNotEmptyConfig(state)) {
                return {...state};
            }

            let blocks = [...state.blocks];
            blocks[action.payload.index] = {...blocks[action.payload.index], ...action.payload.update};

            blocks = updateQuarterBlocks(blocks);

            const shouldUpdateInfo = Boolean(action.payload.shouldUpdateInfo);

            return {
                ...state,
                blocks,
                shouldUpdateInfo,
                shouldSaveInfo: true,
            };
        }
        case QUARTER_CONFIGURATOR_ACTION.ADD_BLOCK:
            if (!checkIsNotEmptyConfig(state)) {
                return {...state};
            }

            return {
                ...state,
                blocks: addQuarterBlock(state.blocks),
                shouldSaveInfo: true,
            };
        case QUARTER_CONFIGURATOR_ACTION.DELETE_BLOCK:
            if (!checkIsNotEmptyConfig(state)) {
                return {...state};
            }

            let blocks = deleteQuarterBlock(state.blocks, action.payload);

            return {
                ...state,
                blocks,
                shouldSaveInfo: true,
            };
        case QUARTER_CONFIGURATOR_ACTION.SET_SHOULD_NOT_UPDATE:
            return {
                ...state,
                shouldUpdateInfo: false,
            };
        case QUARTER_CONFIGURATOR_ACTION.SET_SHOULD_NOT_SAVE:
            return {
                ...state,
                shouldSaveInfo: false,
            };
        case QUARTER_CONFIGURATOR_ACTION.DELETE_SPRINT: {
            if (!checkIsNotEmptyConfig(state)) {
                return {...state};
            }

            const sprints = [...state.sprints].filter((_, index) => index !== action.payload);

            return {
                ...state,
                sprints,
                shouldSaveInfo: true,
            };
        }
        default:
            throw new Error();
    }
};

const getBlocksSumCoefficient = (blocksConfig: Array<QuarterBlock>): number => {
    return +blocksConfig
        .reduce((sum, block) => sum + block.coefficient, 0)
        .toFixed(2);
};

const updateQuarterBlocks = (blocksConfig: Array<QuarterBlock>): Array<QuarterBlock> => {
    const sumCoefficient = getBlocksSumCoefficient(blocksConfig);

    const newBlocksConfig = [...blocksConfig];
    const lastBlockCoefficient = newBlocksConfig[newBlocksConfig.length - 1].coefficient;
    newBlocksConfig[newBlocksConfig.length - 1] = {
        ...newBlocksConfig[newBlocksConfig.length - 1],
        coefficient: (1 - (sumCoefficient - lastBlockCoefficient)) < 0 ?
            0 :
            +(1 - (sumCoefficient - lastBlockCoefficient)).toFixed(2),
    };

    return newBlocksConfig;
};

const deleteQuarterBlock = (blocksConfig: Array<QuarterBlock>, delIndex: number): Array<QuarterBlock> => {
    if (blocksConfig.length < 2) {
        return [...blocksConfig];
    }

    const newBlocksConfig = blocksConfig.filter((_, index) => index !== delIndex);

    return updateQuarterBlocks(newBlocksConfig);
};

const addQuarterBlock = (blocksConfig: Array<QuarterBlock>): Array<QuarterBlock> => {
    const sumCoefficient = getBlocksSumCoefficient(blocksConfig);

    return [...blocksConfig, {
        name: 'Default Name',
        coefficient: (1 - sumCoefficient) < 0 ? 0 : +(1 - sumCoefficient).toFixed(2),
        products: [],
    }];
};

const getQuarterBlockSprint = (
    parentIssueKey = '',
    sprintName = '',
    issues: Record<string, ExtendedIssue> = {},
    issuesToExclude: Set<string> = new Set<string>(),
): {
    capacity: number,
    issues: Array<string>,
} => {
    let quarterBlockSprint = {
        capacity: 0,
        issues: [] as Array<string>,
    };

    if (!issues[parentIssueKey] || sprintName === '') {
        return quarterBlockSprint;
    }

    const parentIssue = issues[parentIssueKey];

    if (!parentIssue.children) {
        if (parentIssue.sprint && parentIssue.sprint[0] && parentIssue.sprint[0].display === sprintName) {
            return {
                capacity: parentIssue.computedStoryPoints,
                issues: [parentIssue.key],
            };
        }

        return quarterBlockSprint;
    }

    const filteredChildren = parentIssue.children.filter((issueKey) => !issuesToExclude.has(issueKey));

    for (const child of filteredChildren) {
        const childQuarterBlockSprint = getQuarterBlockSprint(child, sprintName, issues, issuesToExclude);

        quarterBlockSprint.capacity += childQuarterBlockSprint.capacity;
        quarterBlockSprint.issues = [...quarterBlockSprint.issues, ...childQuarterBlockSprint.issues];
    }

    return quarterBlockSprint;
};

const preformatConfig = (config: QuarterConfiguratorState | {}) => {
    if (!checkIsNotEmptyConfig(config)) {
        return config;
    }

    const {
        blocks,
        issuesMap,
        sprints,
    } = config;

    const issuesToExclude = new Set<string>(
        blocks
            .map((block) => block.products)
            .flat()
            .map(product => product.issueKey)
            .filter(key => typeof key === 'string') as Array<string>
    );

    blocks
        .map((block) => block.products)
        .flat(1)
        .forEach((product) => {
            product.sprints = Object.fromEntries(
                sprints.map((currentSprint) => {
                    const expired = currentSprint.startDate ? new Date() > new Date(currentSprint.startDate) : false;

                    if (!expired) {
                        const currentSprintName = currentSprint.description;
                        const currentSprintCapacity = product.sprints[currentSprintName]?.capacity || 0;

                        return [currentSprintName, {capacity: currentSprintCapacity}];
                    }

                    return [
                        currentSprint.description,
                        getQuarterBlockSprint(product.issueKey, currentSprint.name, issuesMap, issuesToExclude),
                    ];
                }),
            );
        });

    return config;
};


const cn = withNaming({e: '__', m: '_'})('quarter-configurator');

export const QuarterConfigurator: React.VFC<QuarterConfiguratorProps> = ({
                                                                             className,
                                                                             config,
                                                                             setConfig,
                                                                             saveInProgress,
                                                                             updateConfig,
                                                                             updating,
                                                                         }) => {
    const updateSprintHandler = (index: number) => (totalCapacity: string) => {
        dispatch(updateSprintInfoAction({index, update: {totalCapacity: parseFloat(totalCapacity)}}));
    };

    const addSprintHandler = () => dispatch(addSprintAction());

    const sprintDeleteHandler = (index: number) => () => dispatch(deleteSprintAction(index));

    const sprintUpdateHandler = (index: number) => (update: QuarterTableSprintInfo) => {
        dispatch(updateSprintInfoAction({index, update}));
    };

    const getTableData = (sprints: QuarterConfiguratorState['sprints'], config: QuarterConfiguratorState): QuarterTableProps['data'] => {
        const sprintCapacities = sprints.map(sprint => sprint.totalCapacity);

        const plannedCapacities = sprints.map(
            ({description}) =>
                config.blocks.reduce(
                    (sum, {products}) =>
                        products.reduce((blocksSum, {sprints}) => (sprints[description]?.capacity || 0) + blocksSum, 0) + sum, 0,
                ),
        );

        return [
            [
                'Название спринта',
                ...sprints.map(({description, name}, index) => <QuarterTableCellSprint
                    onSprintInfoChange={sprintUpdateHandler(index)}
                    onSprintDelete={sprintDeleteHandler(index)}
                    sprintInfo={{description, name}}
                />),
                <Button
                    size={'s'}
                    view={'clear'}
                    className={cn('add-button')}
                    onClick={addSprintHandler}
                >
                    Добавить
                </Button>,
                'Сумма',
            ],
            [
                'Total SP (capacity)',
                ...sprintCapacities.map((sprintCapacity, index) => <QuarterTableCellEditableText
                    onChange={updateSprintHandler(index)}
                    value={sprintCapacity}
                />),
                undefined,
                <QuarterTableCellEditableText
                    disabled
                    value={sprintCapacities.reduce((sum, currentCapacity) => sum + currentCapacity, 0)}
                />,
            ],
            [
                'Total SP (Planned)',
                ...plannedCapacities.map((totalCapacity) => <QuarterTableCellEditableText
                    disabled
                    value={totalCapacity}
                />),
                '',
                <QuarterTableCellEditableText
                    disabled
                    value={plannedCapacities.reduce((sum, capacity) => sum + capacity, 0)}
                />,
            ],
        ];
    };

    if (!checkIsNotEmptyConfig(config)) {
        config = defaultQuarterConfig;
    }

    const [state, dispatch] = useReducer(reducer, {});
    const [isSyncData, setIsSyncData] = useState(true);
    const [saveChangesMode, setSaveChangesMode] = useState(true);
    const [isMainInfoEditable, setIsMainInfoEditable] = useState(false);

    useEffect(() => {
        dispatch(setConfigAction(preformatConfig(config)));
    }, [config]);

    useEffect(() => {
        if (checkIsNotEmptyConfig(state) && state.shouldUpdateInfo && isSyncData) {
            dispatch(setShouldNotUpdateAction());
            updateConfig(state);
        }
    }, [isSyncData, state, updateConfig]);

    useEffect(() => {
        if (checkIsNotEmptyConfig(state) && state.shouldSaveInfo && saveChangesMode) {
            dispatch(setShouldNotSaveAction());
            setConfig({...state}, true);
        }
    }, [saveChangesMode, setConfig, state]);

    if (!checkIsNotEmptyConfig(state)) {
        return <div className={cn()} />;
    }

    const structure = generateTableStructure(state.sprints.length);
    const data = getTableData(state.sprints, state);

    const commonInfoChangeHandler = () => {
        setIsMainInfoEditable((isEditable) => !isEditable);
    };

    const saveButtonHandler = () => {
        setConfig({...state});
    };

    const quarterNameChangeHandler: EventHandler<ChangeEvent<HTMLInputElement>> = (e) => {
        dispatch(setNameAction(e.target.value));
    };

    const quarterFilterChangeHandler: EventHandler<ChangeEvent<HTMLInputElement>> = (e) => {
        dispatch(setFilterAction(e.target.value));
    };

    const boardChangeHandler = (board: BoardSuggest) => {
        dispatch(setBoardAction(board));
    };

    const deleteBlockHandler = (index: number) => () => dispatch(deleteBlockAction(index));

    const addBlockHandler = () => dispatch(addBlockAction());

    const updateBlockHandler = (index: number) => (update: Partial<QuarterBlock>, shouldUpdateInfo: boolean) => dispatch(
        updateBlockInfoAction({index, update, shouldUpdateInfo}),
    );

    const syncDataButtonClickHandler = () => {
        setIsSyncData(prevState => !prevState);
    };

    const saveDataButtonClickHandler = () => {
        setSaveChangesMode(prevState => !prevState);
    };

    return (
        <div className={cn('', [className])}>
            {
                updating &&
                <div className={cn('paranja')}>
                    <Spin progress view='default' size='m' />
                </div>
            }
            <div className={cn('header')}>
                <div className={cn('header-top-section')}>
                    <div className={cn('header-info')}>
                        {
                            isMainInfoEditable ?
                                <Textinput
                                    theme='normal'
                                    size='m'
                                    autoFocus
                                    value={state.quarterName}
                                    className={cn('title')}
                                    onBlur={quarterNameChangeHandler}
                                /> :
                                <div className={cn('title')}>
                                    {state.quarterName}
                                </div>
                        }
                        <div className={cn('board')}>
                            <Suggest
                                label='Доска'
                                resource='boards'
                                value={state.boardName}
                                onChange={boardChangeHandler}
                            />
                        </div>
                    </div>
                    <div className={cn('actions')}>
                        <Button
                            className={cn('icon')}
                            view={'raised'}
                            onClick={saveButtonHandler}
                            size='s'
                            disabled={saveInProgress}
                        >
                            Сохранить
                        </Button>
                        <Button
                            className={cn('icon', {type: 'save-changes', disabled: !saveChangesMode})}
                            view={'raised'}
                            onClick={saveDataButtonClickHandler}
                            size='s'
                            iconLeft={iconUpload}
                        />
                        <Button
                            className={cn('icon', {type: 'sync', disabled: !isSyncData})}
                            view={'raised'}
                            onClick={syncDataButtonClickHandler}
                            size='s'
                            iconLeft={iconSync}
                        />
                        <Button
                            className={cn('icon')}
                            view={'raised'}
                            onClick={commonInfoChangeHandler}
                            size='s'
                            iconLeft={isMainInfoEditable ? iconConfirm : iconEdit}
                        />
                    </div>
                </div>
                <div className={cn('filter')}>
                    {
                        isMainInfoEditable &&
                        <Textinput
                            theme='normal'
                            size='m'
                            title={'Дополнительный фильтр'}
                            placeholder={'Дополнительный фильтр'}
                            value={state.filter}
                            onBlur={quarterFilterChangeHandler}
                        />
                    }
                    {
                        !isMainInfoEditable && state.filter &&
                        <div>
                            Дополнительный фильтр: <code className={cn('filter-code')}>{state.filter}</code>
                        </div>
                    }
                </div>
            </div>
            <div className={cn('body')}>
                <div className={cn('body-section')}>
                    <QuarterTable
                        data={data}
                        noHeader
                        structure={structure}
                    />
                </div>
                {
                    state.blocks.map((block, index) => (
                        <div key={index} className={cn('body-section')}>
                            <QuarterConfiguratorBlock
                                sprints={state.sprints}
                                issuesMap={state.issuesMap}
                                config={block}
                                onDelete={deleteBlockHandler(index)}
                                coefficientDisabled={index === state.blocks.length - 1}
                                setConfig={updateBlockHandler(index)}
                            />
                        </div>
                    ))
                }
                <div className={cn('body-section')}>
                    <Button
                        className={cn('add-button')}
                        view='raised'
                        width='max'
                        size='s'
                        onClick={addBlockHandler}
                    >
                        add block
                    </Button>
                </div>
            </div>
        </div>
    );
};
