import React, {useRef, useState} from 'react';
import {withNaming} from '@bem-react/classname';
import {Textinput} from '@yandex-lego/components/Textinput/desktop/bundle';
import {Button} from '@yandex-lego/components/Button/desktop/bundle';

import {headerPrefixStructure, suffixStructure, taskPrefixStructure} from './QuarterConfiguratorBlock.const';
import {
    Issue,
    ProductSprintConfig,
    QuarterBlock,
    QuarterBlockProduct,
    QuarterSprint,
} from '../QuarterConfigurator.types';
import {
    QuarterTableCellProduct,
    QuarterTableProductInfo,
} from '../../QuarterTable/QuarterTableCellProduct/QuarterTableCellProduct';
import {QuarterTable, QuarterTableProps} from '../../QuarterTable/QuarterTable';
import {EditIcon} from '../../Icons/EditIcon';
import {ConfirmIcon} from '../../Icons/ConfirmIcon';
import {ColorPicker} from '../../ColorPicker/ColorPicker';
import {StickerIcon} from '../../Icons/StickerIcon';
import {TrashIcon} from '../../Icons/TrashIcon';
import {presentData} from './QuarterConfiguratorBlock.utils';
import {QuarterTableRowProps} from '../../QuarterTable/QuarterTableRow/QuarterTableRow';
import {QuarterTableCellColorPicker} from '../../QuarterTable/QuarterTableCellColorPicker/QuarterTableCellColorPicker';
import {
    QuarterTableCellEditableText,
} from '../../QuarterTable/QuarterTableCellEditableText/QuarterTableCellEditableText';
import {
    QuarterTableCellProductCapacity,
} from '../../QuarterTable/QuarterTableCellProductCapacity/QuarterTableCellProductCapacity';
import {RightPanel} from '../../RightPanel/RightPanel';
import {TaskFrame} from '../../TaskFrame/TaskFrame';

import './QuarterConfiguratorBlock.css';


interface QuarterConfiguratorBlockProps {
    sprints: Array<QuarterSprint>;
    config: QuarterBlock;
    issuesMap: Record<string, Issue>;

    setConfig(config: Partial<QuarterBlock>, shouldUpdateInfo?: boolean): void;

    className?: string;
    coefficientDisabled?: boolean;

    onDelete?(): void;
}

const generateTableStructure = (
    prefix: QuarterTableRowProps['structure'],
    suffix: QuarterTableRowProps['structure'],
    sprintsNumber: number,
    backgroundColor?: string,
) => {
    const sprintsStructure: QuarterTableRowProps['structure'] = [{
        backgroundColor: backgroundColor || 'rgb(255, 234, 158)',
        cells: Array.from({length: sprintsNumber}, () => ({
            type: 'default',
        })),
    }];

    return prefix.concat(sprintsStructure, suffix);
};

const cn = withNaming({e: '__', m: '_'})('quarter-configurator-block');

const iconEdit = (className: string) => <EditIcon className={className} />;
const iconConfirm = (className: string) => <ConfirmIcon className={className} />;
const iconColorPicker = (className: string) => <StickerIcon className={className} />;
const iconDelete = (className: string) => <TrashIcon className={className} />;

const getSprintCapacity = (sprints: Record<string, ProductSprintConfig>, description = '') => {
    return sprints[description]?.capacity || 0;
};

const getHeaderTableData = (config: QuarterBlock, sprints: QuarterConfiguratorBlockProps['sprints']): QuarterTableProps['data'] => {
    const plannedSp = sprints.map(({description}) =>
        config.products.reduce((sumCoefficient, {sprints}) =>
            getSprintCapacity(sprints, description) + sumCoefficient, 0),
    );

    return [
        [
            'Название спринта',
            ...sprints.map(({description}) => description),
            undefined,
            'Сумма',
        ],
        [
            'Даты окончания спринта',
            ...sprints.map(({endDate}) => presentData(endDate)),
            undefined,
        ],
        [
            'Предел SP',
            ...sprints.map(({totalCapacity}) => (
                <QuarterTableCellEditableText
                    disabled
                    value={Math.round(10 * config.coefficient * totalCapacity) / 10}
                />
            )),
            '',
            <QuarterTableCellEditableText
                disabled
                value={Math.round(10 * sprints.reduce((sum, sprint) => sprint.totalCapacity + sum, 0) * config.coefficient) / 10}
            />,
        ],
        [
            'Сколько напланировано',
            ...plannedSp.map((currentSprintPlannedSp) => (
                <QuarterTableCellEditableText
                    disabled
                    value={currentSprintPlannedSp}
                />
            )),
            '',
            <QuarterTableCellEditableText
                disabled
                value={plannedSp.reduce((sum, currentSprintPlannedSp) => sum + currentSprintPlannedSp, 0)}
            />,
        ],
    ];
};

export const QuarterConfiguratorBlock: React.VFC<QuarterConfiguratorBlockProps> = ({
                                                                                       className,
                                                                                       setConfig,
                                                                                       config,
                                                                                       sprints,
                                                                                       coefficientDisabled,
                                                                                       onDelete,
                                                                                       issuesMap,
                                                                                   }) => {
    const [isVisible, setIsVisible] = useState(false);
    const [isMainInfoEditable, setIsMainInfoEditable] = useState(false);
    const [taskPanelVisible, setTaskPanelVisible] = useState(false);
    const [selectedTask, setSelectedTask] = useState('');

    const ref = useRef<HTMLElement>(null);

    const editColorButtonClickHandler = () => setIsVisible((prevState) => !prevState);
    const editButtonClickHandler = () => setIsMainInfoEditable((isEditable) => !isEditable);

    const hideTask = () => setTaskPanelVisible(false);

    const issueSelectHandler = (issueKey: string) => {
        setSelectedTask(issueKey);
        setTaskPanelVisible(true);
    };

    const productBudgetChangeHandler = (index: number) => (budget: string) => {
        const {products} = config;
        products[index].budget = parseFloat(budget || '0');

        setConfig({products});
    };

    const productCapacityChangeHandler = (index: number, sprint: string) => (capacity: string) => {
        const {products} = config;
        const parsedCapacity = parseFloat(capacity);

        if (!sprint) {
            return;
        }

        if (products[index].sprints[sprint]) {
            products[index].sprints[sprint].capacity = parsedCapacity;
        } else {
            products[index].sprints[sprint] = {capacity: parsedCapacity, issues: []};
        }

        setConfig({products});
    };

    const productInfoChangeHandler = (index: number) => ({name, issueKey, weight}: QuarterTableProductInfo) => {
        const {products} = config;
        const shouldUpdateInfo = products[index].issueKey !== issueKey;

        products[index] = {...products[index], name, issueKey, weight};

        setConfig({products}, shouldUpdateInfo);
    };

    const productDeleteHandler = (index: number) => () => {
        const filteredProducts = config.products.filter((_, productIndex) => productIndex !== index);

        setConfig({products: filteredProducts});
    };

    const colorChangeHandler = (index: number) => (color: string) => {
        const {products} = config;
        products[index].color = color;

        setConfig({products});
    };

    const coefficientChangeHandler: React.ChangeEventHandler<HTMLInputElement> = (e) => {
        const coefficient = parseFloat(e.target.value) || 0;

        if (coefficient < 0) {
            return setConfig({coefficient: 0});
        }

        if (coefficient > 1) {
            return setConfig({coefficient: 1});
        }

        setConfig({coefficient});
    };

    const blockNameChangeHandler: React.ChangeEventHandler<HTMLInputElement> = (e) => {
        const name = e.target.value;
        setConfig({name});
    };

    const addProductButtonClickHandler = () => {
        const newProduct: QuarterBlockProduct = {
            budget: 0,
            name: 'new product',
            sprints: {},
        };

        setConfig({products: config.products.concat(newProduct)});
    };

    const matrixColorChangeHandler = (color: string | undefined) => {
        setConfig({color});
    };

    const getProductsTableData = (config: QuarterBlock, sprints: QuarterConfiguratorBlockProps['sprints']): QuarterTableProps['data'] => {
        return [
            [
                'Продукт',
                'Цвет',
                'Бюджет',
                ...sprints.map(({description}) => description),
                undefined,
                'Сумма',
            ],
            ...config.products
                .sort((a, b) => (b.weight || 0) - (a.weight || 0))
                .map((product, index) => ({
                    backgroundCellColor: product.color, values: [
                        <QuarterTableCellProduct
                            key={`${product.issueKey}_${index}`}
                            onDelete={productDeleteHandler(index)}
                            onIssueLinkClick={issueSelectHandler}
                            productInfo={{
                                name: product.name,
                                issueKey: product.issueKey,
                                description: issuesMap[product.issueKey || '']?.summary,
                                weight: product.weight,
                            }}
                            onProductInfoChange={productInfoChangeHandler(index)}
                        />,
                        <QuarterTableCellColorPicker
                            onChange={colorChangeHandler(index)}
                            selectedColor={product.color}
                        />
                        ,
                        <QuarterTableCellEditableText
                            value={product.budget}
                            onChange={productBudgetChangeHandler(index)}
                        />,
                        ...sprints.map(
                            ({description, startDate}) => {
                                const expired = startDate ? new Date() > new Date(startDate) : false;

                                return (
                                    <QuarterTableCellProductCapacity
                                        expired={expired}
                                        value={getSprintCapacity(product.sprints, description)}
                                        issuesMap={issuesMap}
                                        issuesKeys={product.sprints[description]?.issues || []}
                                        onChange={productCapacityChangeHandler(index, description)}
                                    />
                                );
                            },
                        ),
                        '',
                        <QuarterTableCellEditableText
                            disabled
                            value={sprints.reduce((sum, {description}) => sum + getSprintCapacity(product.sprints, description), 0)}
                        />,
                    ],
                })),
        ];
    };

    return (
        <div className={cn('', [className])}>
            {taskPanelVisible && (
                <RightPanel onClose={hideTask}>
                    <TaskFrame taskKey={selectedTask} />
                </RightPanel>
            )}
            <div className={cn('header-info-wrapper')}>
                <div className={cn('header-info')}>
                    {
                        isMainInfoEditable ?
                            <Textinput
                                className={cn('title')}
                                theme='normal'
                                size='m'
                                value={config.name}
                                onBlur={blockNameChangeHandler}
                            /> :
                            <div className={cn('title')}>
                                {config.name}
                            </div>
                    }
                    {
                        isMainInfoEditable && !coefficientDisabled ?
                            <Textinput
                                className={cn('coefficient')}
                                theme='normal'
                                size='m'
                                value={config.coefficient}
                                inputMode={'numeric'}
                                onBlur={coefficientChangeHandler}
                            /> :
                            <div className={cn('coefficient')}>
                                Коэффициент: {config.coefficient}
                            </div>
                    }
                </div>
                <div className={cn('actions')}>
                    <div style={{fill: config.color}}>
                        <Button
                            className={cn('icon')}
                            view={'raised'}
                            onClick={editColorButtonClickHandler}
                            innerRef={ref}
                            size='s'
                            iconLeft={iconColorPicker}
                        />
                        <ColorPicker
                            anchor={ref}
                            visible={isVisible}
                            selectedColor={config.color}
                            onChange={matrixColorChangeHandler}
                        />
                    </div>
                    {
                        isMainInfoEditable && (!coefficientDisabled || !config.coefficient) &&
                        <Button
                            className={cn('icon')}
                            view={'raised'}
                            onClick={onDelete}
                            size='s'
                            iconLeft={iconDelete}
                        />
                    }
                    <Button
                        className={cn('icon')}
                        view={'raised'}
                        onClick={editButtonClickHandler}
                        size='s'
                        iconLeft={isMainInfoEditable ? iconConfirm : iconEdit}
                    />
                </div>
            </div>

            <div className={cn('body')}>
                <div className={cn('body-section')}>
                    <QuarterTable
                        data={getHeaderTableData(config, sprints)}
                        noHeader
                        structure={generateTableStructure(
                            headerPrefixStructure,
                            suffixStructure,
                            sprints.length,
                            config.color,
                        )}
                    />
                </div>
                <div className={cn('body-section')}>
                    <QuarterTable
                        data={getProductsTableData(config, sprints)}
                        structure={generateTableStructure(
                            taskPrefixStructure,
                            suffixStructure,
                            sprints.length,
                            config.color,
                        )}
                    />
                </div>
            </div>
            <div className={cn('body-section', {'with-actions': true})}>
                <Button
                    className={cn('add-button')}
                    size={'s'}
                    view={'raised'}
                    onClick={addProductButtonClickHandler}
                >
                    Добавить строку
                </Button>
            </div>
        </div>
    );
};
