import React, {useState} from 'react';
import {withNaming} from '@bem-react/classname';
import {Textinput} from '@yandex-lego/components/Textinput/desktop/bundle';
import {Link as LegoLink} from '@yandex-lego/components/Link/desktop/bundle';
import {Button} from '@yandex-lego/components/Button/desktop/bundle';

import {Suggest} from '../../Suggest/Suggest';

import './QuarterTableCellProduct.css';


export interface QuarterTableProductInfo {
    name: string;
    description?: string;
    issueKey?: string;
    weight?: number;
}

interface QuarterTableCellProductProps {
    productInfo: QuarterTableProductInfo;

    onProductInfoChange(productInfo: QuarterTableProductInfo): void;

    onDelete(): void;

    onIssueLinkClick(issueId?: string): void;

    className?: string;
}

const cn = withNaming({e: '__', m: '_'})('quarter-table-cell-product');

export const QuarterTableCellProduct: React.VFC<QuarterTableCellProductProps> = ({
                                                                                     className,
                                                                                     onProductInfoChange,
                                                                                     productInfo: {
                                                                                         weight,
                                                                                         name,
                                                                                         description,
                                                                                         issueKey,
                                                                                     },
                                                                                     onIssueLinkClick,
                                                                                     onDelete,
                                                                                 }) => {
    const [isEditMode, setIsEditMode] = useState(false);

    const editButtonClickHandler = () => {
        setIsEditMode(isEditMode => !isEditMode);
    };

    const issueLinkClickHandler = () => onIssueLinkClick(issueKey);

    const productNameChangeHandler: React.ChangeEventHandler<HTMLInputElement> = (e) => {
        onProductInfoChange({
            name: e.target.value,
            weight,
            issueKey,
        });
    };

    const productWeightChangeHandler: React.ChangeEventHandler<HTMLInputElement> = (e) => {
        onProductInfoChange({
            issueKey,
            weight: parseFloat(e.target.value) || 0,
            name,
        })
    }

    const issueChangeHandler = (issue: {key: string}) => {
        onProductInfoChange({
            issueKey: issue.key,
            weight,
            name,
        });
    };

    return (
        <div className={cn('', [className])}>
            <div className={cn('header')}>
                {
                    isEditMode ?
                        <Textinput
                            className={cn('name')}
                            theme='normal'
                            size='s'
                            value={name}
                            autoFocus
                            onBlur={productNameChangeHandler}
                        /> :
                        <div className={cn('name')}>
                            {name}
                        </div>
                }
                {
                    isEditMode ?
                        <Textinput
                            className={cn('weight')}
                            theme='normal'
                            size='s'
                            placeholder={'вес'}
                            value={weight}
                            onBlur={productWeightChangeHandler}
                        /> :
                        <div className={cn('weight-text')}>
                            {weight}
                        </div>
                }
                <Button
                    onClick={editButtonClickHandler}
                    className={cn('edit-button')}
                    view={'clear'}
                >
                    {isEditMode ? 'Принять' : 'Изменить'}
                </Button>
                <Button
                    onClick={onDelete}
                    className={cn('delete-button')}
                    view={'clear'}
                >
                    Удалить
                </Button>
            </div>
            <div className={cn('product-task')}>
                {
                    isEditMode ?
                        <Suggest
                            resource='issues'
                            value={issueKey}
                            onChange={issueChangeHandler}
                        /> :
                        <LegoLink
                            target='_blank'
                            theme='ghost'
                            title={description}
                            onClick={issueLinkClickHandler}
                        >
                            {issueKey}
                        </LegoLink>
                }
            </div>
        </div>
    );
};
