import React, {useRef, useState} from 'react';
import {withNaming} from '@bem-react/classname';
import {Tooltip} from '@yandex-lego/components/Tooltip/desktop/bundle';
import {Textinput} from '@yandex-lego/components/Textinput/desktop/bundle';
import {Button} from '@yandex-lego/components/Button/desktop/bundle';

import {useClickOutside} from '../hooks';
import {Suggest} from '../../Suggest/Suggest';
import {QuarterTableSprintInfo} from '../../QuarterConfigurator/QuarterConfigurator.types';

import './QuarterTableCellSprintSelect.css';

interface QuarterTableCellSprintProps {
    sprintInfo: QuarterTableSprintInfo;

    onSprintInfoChange(sprintInfo: QuarterTableSprintInfo): void;

    onSprintDelete(): void;

    className?: string;
    suffix?: string;
}

const cn = withNaming({e: '__', m: '_'})('quarter-table-cell-sprint-select');

export const QuarterTableCellSprint: React.VFC<QuarterTableCellSprintProps> = ({
                                                                                   className,
                                                                                   onSprintInfoChange,
                                                                                   sprintInfo: {
                                                                                       description,
                                                                                       name,
                                                                                   },
                                                                                   suffix,
                                                                                   onSprintDelete,
                                                                               }) => {
    const [isEditMode, setIsEditMode] = useState(false);
    const cellRef = useRef<HTMLDivElement>(null);
    const tooltipRef = useRef<HTMLDivElement>(null);

    const clickOutsideHandler = React.useCallback(() => {
        setIsEditMode(false);
    }, []);

    const sprintChangeHandler = (sprint?: {id: number, name: string, endDate: string}) => {
        if (!sprint) {
            return onSprintInfoChange({
                id: undefined,
                name: undefined,
                description,
                endDate: undefined
            });
        }

        onSprintInfoChange({
            id: sprint.id,
            name: sprint.name,
            description,
            endDate: sprint.endDate
        });
    };

    const sprintDescriptionChangeHandler: React.ChangeEventHandler<HTMLInputElement> = (e) => {
        onSprintInfoChange({description: e.target.value});
    };

    const cellClickHandler = () => {
        setIsEditMode(true);
    }

    useClickOutside(tooltipRef, clickOutsideHandler);

    return (
        <div
            ref={cellRef}
            onClick={cellClickHandler}
            className={cn('', [className])}
        >
            {description} {suffix}
            <Tooltip
                hasTail
                direction='bottom'
                view='default'
                size='l'
                className={cn('tooltip')}
                innerRef={tooltipRef}
                anchor={cellRef}
                visible={isEditMode}
            >
                <Textinput
                    autoFocus
                    size='s'
                    view='default'
                    theme='normal'
                    value={description}
                    className={cn('description')}
                    onBlur={sprintDescriptionChangeHandler}
                />
                <Suggest
                    label='Спринт'
                    resource='sprints'
                    value={name}
                    valueProp={'name'}
                    additionalProps={['board']}
                    onChange={sprintChangeHandler}
                />
                <Button
                    view={'raised'}
                    size="s"
                    onClick={onSprintDelete}
                    className={cn('delete-button')}
                >
                    Удалить
                </Button>
            </Tooltip>
        </div>
    );
};
